<?php
/**
 * Despatches orders using the fulfillments end point.
 *
 * System/Version:  Shopify
 * Developer:       Natasha Mitchell
 * OnTime Ref:      62327
 * Completed:       28/03/2023
 * Production URL:  http://cestevenson-orderwise.net/orderwise_shopify/despatch.php
 */

/** @var Settings $settings */
/** @var Shopify $shopify */
/** @var Handler $ecom */
/** @var DataFormat $format */
require __DIR__ . '/config.php';

// Handle the POST data
$export_data = isset($_POST['ExportData']) ? stripslashes(trim($_POST['ExportData'])) : null;
if ($export_data == '') {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_WARNING, 'No export data to process');
    $ecom->ReturnResponseXML();
}
$ecom->ArchiveOrderwiseExportedData('despatch', $export_data);
if (!$xml = @simplexml_load_string($export_data)) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, 'Exported XML invalid');
    $ecom->ReturnResponseXML();
}
if (count($xml) == 0) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_WARNING, 'No records exported');
    $ecom->ReturnResponseXML();
}

// Instantiate Shopify validate the store id from the url
try {
    $shopify = new Shopify($ecom->GetStoreId(), $settings, $ecom, $format);
} catch (Exception $e) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
    $ecom->ReturnResponseXML();
}

// Work through the XML and build a shipments array, grouping the lines by order ID.
$updated = 0;
if (count($xml) > 0) {
    $shipments = [];
    $tracking_numbers = [];
    $fulfillment_order_details = [];
    foreach ($xml as $x) {
        $order_id = trim($x->order_id);
        $order_item_code = trim($x->order_item_code);
        $tracking_number = (string) $x->shipping_tracking_number;
        $shipping_carrier = trim($x->shipping_carrier);
        $tracking_link = $shopify->getCourierLink($shipping_carrier);
        $quantity = floor(trim($x->quantity_despatched));

        if ($order_item_code == '') {
            $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, 'Order item code not set for order ID ' . $order_id);

            continue;
        }

        // Get the fulfillment order details for the order from Shopify so we have the detail to despatch the order.
        try {
            if (empty($fulfillment_order_details[$order_id])) {
                $fulfillment_order_details[$order_id] = $shopify->fetchFulfilmentOrderDetails($order_id, 'despatch');
            }
        } catch (Exception $e) {
            $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());

            continue;
        }

        if (!array_key_exists($order_item_code, $fulfillment_order_details[$order_id]['fulfillment_lines'])) {
            $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, 'Order item code ' . $order_item_code . ' doesn\'t exist on Shopify order ID ' . $order_id);

            continue;
        }

        if (array_key_exists($order_item_code, $fulfillment_order_details[$order_id]['lines'])) {
            $shopify_stock_location_id = (string) $fulfillment_order_details[$order_id]['lines'][$order_item_code];
        } else {
            $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, 'Stock location information not found for ' . $order_item_code . 'on Shopify order ID ' . $order_id);

            continue;
        }

        if (empty($tracking_numbers[$order_id][$shopify_stock_location_id][$tracking_number])) {
            // tracking number doesn't exist, set it against the array now so we can't add it again later
            $tracking_numbers[$order_id][$shopify_stock_location_id][$tracking_number] = true;
        } else {
            // tracking number already exists, set it to null so we don't set it again in Shopify
            $tracking_number = null;
        }

        // Create the tracking URL using the csv file.
        $link = null;
        if (!is_null($tracking_number)) {
            $link = str_ireplace('[tracking_code]', trim($tracking_number), $tracking_link);
        }

        if (empty($shipments[$order_id])) {
            $shipments[$order_id] = [];
        }

        if (empty($shipments[$order_id][$shopify_stock_location_id])) {
            $shipments[$order_id][$shopify_stock_location_id] = [];
        }

        $shipments[$order_id][$shopify_stock_location_id][] = [
            'company' => $shipping_carrier,
            'order_item_code' => $order_item_code,
            'tracking' => $tracking_number ?: null,
            'tracking_url' => $link,
            'quantity' => $quantity,
            'stock_location' => $shopify_stock_location_id,
        ];
    }

    // Create the fulfillment array using the details from the XML, and the information from the fulfillment order details from Shopify.
    foreach ($shipments as $order_id => $despatch_location) {
        $shipment = [];
        $lines = [];
        foreach ($despatch_location as $tracking) {
            foreach ($tracking as $line) {
                $fulfillment_order_id = $fulfillment_order_details[$order_id]['fulfillment_lines'][$line['order_item_code']]['fulfillment_order_id'];
                $fulfillment_line_id = $fulfillment_order_details[$order_id]['fulfillment_lines'][$line['order_item_code']]['fulfillment_line_id'];

                // Add any quantities for the lines together to prevent duplicate ID's being pushed to Shopify.
                if (isset($lines[$fulfillment_order_id][$fulfillment_line_id])) {
                    $lines[$fulfillment_order_id][$fulfillment_line_id] += $line['quantity'];
                } else {
                    $lines[$fulfillment_order_id][$fulfillment_line_id] = $line['quantity'];
                }

                $shipment[$fulfillment_order_id]['fulfillment']['line_items_by_fulfillment_order'][0]['fulfillment_order_id'] = $fulfillment_order_id;
                $shipment[$fulfillment_order_id]['fulfillment']['notify_customer'] = 1;

                // Populate the origin address based on the stock location exported. This doesn't determine the stock location it is shipped from in Shopify, but
                // this will populate the origin address information against the fulfillment.
                if (!isset($shipment[$fulfillment_order_id]['fulfillment']['origin_address'])) {
                    $shipment[$fulfillment_order_id]['fulfillment']['origin_address'] = [
                        'address1' => $fulfillment_order_details[$order_id]['assigned_locations'][$line['stock_location']]->address1,
                        'address2' => $fulfillment_order_details[$order_id]['assigned_locations'][$line['stock_location']]->address2,
                        'city' => $fulfillment_order_details[$order_id]['assigned_locations'][$line['stock_location']]->city,
                        'country_code' => $fulfillment_order_details[$order_id]['assigned_locations'][$line['stock_location']]->country_code,
                        'province_code' => $fulfillment_order_details[$order_id]['assigned_locations'][$line['stock_location']]->province_code,
                        'zip' => $fulfillment_order_details[$order_id]['assigned_locations'][$line['stock_location']]->zip,
                    ];
                }

                // Where it's set, populate the tracking information against the fulfillment.
                if (isset($line['company'])) {
                    $shipment[$fulfillment_order_id]['fulfillment']['tracking_info']['company'] = $line['company'];
                }

                if (isset($line['tracking'])) {
                    $shipment[$fulfillment_order_id]['fulfillment']['tracking_info']['number'] = $line['tracking'];
                }

                if (isset($line['tracking_url'])) {
                    $shipment[$fulfillment_order_id]['fulfillment']['tracking_info']['url'] = $line['tracking_url'];
                }
            }
        }

        // Ship each fulfillment order for the order ID exported.
        foreach ($shipment as $fulfilment_order_id => $fulfillment) {
            // Add the lines to the block before shipping.
            foreach($lines as $line_order_id => $line) {
                foreach($line as $line_id => $quantity) {
                    if ($line_order_id == $fulfilment_order_id) {
                        $fulfillment['fulfillment']['line_items_by_fulfillment_order'][0]['fulfillment_order_line_items'][] = [
                            'id' => $line_id,
                            'quantity' => $quantity,
                        ];
                    }
                }
            }

            try {
                $shopify->SaveShipment($order_id, $fulfillment);
                $updated++;
            } catch (Exception $e) {
                $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
            }
        }
    }
}

// Send any response messages back
$ecom->AppendResponseMessage(XML_REPONSE_TYPE_LOG, 'Updated ' . $updated . ' record(s)');
$ecom->ReturnResponseXML();
