<?php

class Shopify
{
    const ORDER_PAGINATION_LIMIT = 100;
    const PRODUCT_PAGINATION_LIMIT = 250;
    const STOCK_LOCATION_PAGINATION_LIMIT = 50;
    public $stock_locations = [];
	public $import_as_gross = true; // whether or not orders should be imported using the gross values or not
    private $_api_end_point;
    // 17/10/2018 | AC | Dev ID 39000 | New array for use in the getTaxTitles method
    //21/09/2020  | TL | Bug ID 67652 | Added Shipping VAT
    private $_tax_titles = [
        'VAT' => 'VAT',
        'Shipping VAT' => 'Shipping VAT'  // 21/09/2020 | TL | Bug 67652 | Shipping VAT
    ];
    public $import_discount_as_gross;

    //  Dev 47605 - below added for cursor pagination changes
    const API_VERSION = '2024-07';
    private $_current_cursor_pagination_urls = [
        'orders' => ['previous' => '', 'next' => '', 'is_next_page_available' => true],
        'stock' => ['previous' => '', 'next' => '', 'is_next_page_available' => true],
        'price' => ['previous' => '', 'next' => '', 'is_next_page_available' => true],
        'barcodes' => ['previous' => '', 'next' => '', 'is_next_page_available' => true], // 13/07/2020 | CB | Dev 49316 | Added
        'stock_locations' => ['previous' => '', 'next' => '', 'is_next_page_available' => true], // 17/02/2023 | NM | Dev 61577 | Added
    ];


    /**
     * @param $store_id
     * @param Settings $settings
     * @throws Exception
     */
    function __construct($store_id, $settings = null)
    {
        $stores = [];

        // Main Store
        $stores[1] = [
            'key' => '3ab1e7e213c7d3cb4fcc8274327c81b1',
            'password' => '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',
            'shoppath' => 'hyfive-products.myshopify.com',
        ];

        if (!isset($stores[$store_id])) {
            throw new Exception('Invalid store id in URL.');
        } else {
            if (!in_array("", $stores[$store_id], true)) {
                // CB | Dev 47605 | Force specific version of API
                $this->_api_end_point = 'https://' . $stores[$store_id]['key'] . ':' . $stores[$store_id]['password'] . '@' . $stores[$store_id]['shoppath'] . '/admin/api/' . self::API_VERSION . '/';
            } else {
                throw new Exception("Store does not have fully configured data for store #{$store_id}. Please make sure you passed the correct store ID in the URL.");
            }
        }

        // 23/01/2020 | TL | Dev ID 48002 | Sleep for a second to prevent more than 1 calls a second.
        usleep(600);
        //22/05/2019 | TL | Dev 43368 | Script Edit to handle multiple stock locations | call and store stock location information.
        $this->stock_locations = $this->GetStockLocations();

        // load saved script-specific settings from the settings manager
        if (isset($settings)) {
            // import as gross not fully setup and would need testing
            $this->import_as_gross = $settings->GetScriptOption('import_as_gross');
            $this->import_discount_as_gross = $settings->GetScriptOption('import_discount_as_gross');
        }
    }


    public function GetStockLocations()
    {
        $stock_locations = [];
        do {
            // Make the request
            $response = $this->_ApiCall(['url' => 'locations.json?limit=' . self::STOCK_LOCATION_PAGINATION_LIMIT], 'stock_locations');

            // Decode the response
            $r = json_decode($response['body']);

            if (isset($r->errors)) {
                $msg = $this->_GetErrorMessage(json_decode($response['body']));

                throw new Exception('API Error: Unable to retrieve stock locations ' . $msg ? $msg : $response);
            }

            foreach ($r->locations as $location) {
                $stock_locations[(string) $location->id] = $location;
            }
        } while ($this->isNextPageAvailable('stock_locations'));

        return $stock_locations;
    }


    private function _ApiCall($data, $pagination_cursor_type = null, $retried = false)
    {
        // 12/02/2020 | CB | Dev 47605 | Build up url
        $url = $this->_api_end_point . $data['url'];

        // 12/02/2020 | CB | Dev 47605 | Add cursor pagination info where required
        if (!empty($pagination_cursor_type)) {
            $url = $this->getNextPageUrl($pagination_cursor_type, $url);
        }

        if (empty($url) || $url == $this->_api_end_point) {
            return false;
        }

        $ch = curl_init($url);

        // Set the defaults for a basic connection
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, true);

        // Override the defaults and add any additional values
        if (is_array($data['config'])) {
            foreach ($data['config'] as $k => $v) {
                curl_setopt($ch, $k, $v);
            }
        }
        $r = curl_exec($ch);

        // Run the curl request
        $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $response = [
            'content' => $r,
            'header' => substr($r, 0, $header_size),
            'body' => substr($r, $header_size),
            'http_code' => curl_getinfo($ch, CURLINFO_HTTP_CODE),
            'header_size' => $header_size,
        ];
        curl_close($ch);

        // Max calls is 2 per second. Sleep for .6 of a second to keep within their call limits.
        usleep(600);

        if (!$retried && empty($data['config'][CURLOPT_POST])) {
            usleep(5000); // the request failed -- wait a longer period of time before trying again (GET requests only)

            return $this->_ApiCall($data, $pagination_cursor_type, true);
        }

        // 12/02/2020 | CB | Dev 47605 | Get cursor pagination info where possible
        if (!empty($pagination_cursor_type)) {
            $this->setCursorPaginationLinks($pagination_cursor_type, $response);
        }

        return $response;
    }


    private function _GetErrorMessage($obj)
    {
        $errors = [];
        if (isset($obj->errors)) {
            if (!is_object($obj->errors) && !is_array($obj->errors)) {
                return trim($obj->errors);
            }
            foreach ($obj->errors as $field => $v) {
                if (!is_array($v)) {
                    $errors[] = $field . ': ' . urldecode($v);

                    continue;
                }

                foreach ($v as $error) {
                    $errors[] = $field . ': ' . urldecode($error);
                }
            }
        }

        return implode(' | ', $errors);
    }


    public function getCourierLink($courier)
    {
        $csv = array_map('str_getcsv', file('courier_link.csv'));
        $link = '';

        for ($i = 0; $i < count($csv); $i++) {
            if (strtolower($csv[$i][0]) == strtolower($courier)) {
                $link = $csv[$i][1];
            }
        }

        if (empty($link)) {
            $link = 'http://www.google.com/search?q=[tracking_code]';
        }

        return $link;
    }


    public function LoadOrders($filter)
    {
        // If the filter contains an order id, this will exclude
        // any other filters being passed in order to pull the order
        if (is_numeric($filter['order_id'])) {
            // Make the request

            $url = str_replace('{id}', $filter['order_id'], 'orders/{id}.json');
            $response = $this->_ApiCall(['url' => $url]);

            // Decode the response
            $r = json_decode($response['body']);
            if (isset($r->errors)) {
                $msg = $this->_GetErrorMessage(json_decode($response['body']));
                throw new Exception('API Error: Unable to retrieve order ' . $filter['order_id'] . ' [' . $msg ? $msg : $response . ']');
            }

            // When returning orders from a general call an array is returned,
            // so we mimic this to ensure the same code can process the response
            return [$r->order];
        } // Loading orders based on filtering
        else {
            if (count($filter) == 0) {
                return;
            }

            if (!$this->isNextPageAvailable('orders'))
            {
                return [];
            }

            $url = 'orders.json?limit=' . self::ORDER_PAGINATION_LIMIT;

            // If you pass any other params but limit once you are paging through,
            // Shopify will throw an error rather than handle/ignore these
            if ($this->isFirstPageCall('orders')) {
                foreach ($filter as $k => $v) {
                    if ($v != '') {
                        $url .= '&' . $k . '=' . $v;
                    }
                }
            }

            $response = $this->_ApiCall(['url' => $url], 'orders');

            if (empty($response))
            {
                return [];
            }

            $r = json_decode($response['body']);

            if (isset($r->errors)) {
                $msg = $this->_GetErrorMessage(json_decode($response['body']));
                throw new Exception('API Error: Unable to retrieve orders:  ' . $msg ? $msg : $response);
            }

            // Return the orders array
            return $r->orders;
        }
    }


    public function fetchFulfilmentOrderDetails($order, $type = 'order')
    {
        if ($type == 'order' && empty($this->load_order_line_fulfilment_details)) {
            return [];
        }

        $order_id = $type == 'despatch' ? $order : $order->id;

        $fulfilment_order_details = [
            'assigned_locations' => [],
            'lines' => [],
            'fulfillment_lines' => [],
        ];

        $url = str_replace('{id}', $order_id, 'orders/{id}/fulfillment_orders.json');
        $response = $this->_ApiCall(['url' => $url]);

        // Decode the response
        $r = json_decode($response['body']);

        if (isset($r->errors)) {
            $msg = $this->_GetErrorMessage(json_decode($response['body']));

            throw new Exception('API Error: Unable to retrieve order fulfilment details ' . $order_id . ' [' . $msg ? $msg : $response . ']');
        }

        if (empty($r->fulfillment_orders)) {
            if ($type == 'despatch') {
                throw new Exception('No order fulfillment details returned from Shopify for order ID ' . $order_id);
            }

            return [];
        }

        foreach ($r->fulfillment_orders as $fulfilment_data) {
            if (empty($fulfilment_data->line_items)) {
                continue;
            }

            if (empty($fulfilment_data->assigned_location_id)) {
                continue;
            }

            if (empty($fulfilment_data->assigned_location)) {
                continue;
            }

            $location_id = (string)$fulfilment_data->assigned_location_id;

            // If for any reason the stock location id is no longer in our list, used the details against the order fulfilment data
            $fulfilment_order_details['assigned_locations'][$location_id] = (empty($this->stock_locations[$location_id]))
                ? $fulfilment_data->assigned_location
                : $this->stock_locations[$location_id];

            foreach ($fulfilment_data->line_items as $fulfilment_line_data) {
                $fulfilment_order_details['lines'][(string)$fulfilment_line_data->line_item_id] = $location_id;
                $fulfilment_order_details['fulfillment_lines'][(string)$fulfilment_line_data->line_item_id]['fulfillment_order_id'] = (string)$fulfilment_data->id;
                $fulfilment_order_details['fulfillment_lines'][(string)$fulfilment_line_data->line_item_id]['fulfillment_line_id'] = (string)$fulfilment_line_data->id;
            }
        }

        return $fulfilment_order_details;
    }


    /**
     * Gets an order's transactions records. This may contain multiple
     * records as any failed payment attempts are also included
     */
    public function GetTransaction($order_id)
    {
        // Make the request
        // 22/11/2019 | CB | Dev 46743 | Download payment in shop currency
        $url = str_replace('{id}', $order_id, 'orders/{id}/transactions.json?in_shop_currency=true');
        $response = $this->_ApiCall(['url' => $url]);

        // Decode the response
        $r = json_decode($response['body']);
        if (isset($r->errors)) {
            $msg = $this->_GetErrorMessage(json_decode($response['body']));
            throw new Exception('API Error: Unable to retrieve order transaction ' . $order_id . ' [' . $msg ? $msg : $response . ']');
        }

        return $r->transactions;
    }


    public function SaveShipment($order_id, $data)
    {
        // All submissions must be in json
        $json_data = json_encode($data);

        // Configure curl
        $api = [
            'url' => 'fulfillments.json',
            'config' => [
                CURLOPT_HEADER => true,
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => $json_data,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/json',
                    'Content-Length: ' . strlen($json_data),
                ],
            ],
        ];

        // Make the api request
        $response = $this->_ApiCall($api);

        // Throw an exception if we have errors
        if ($response['http_code'] != 201 && $response['http_code'] != 200) {
            $msg = $this->_GetErrorMessage(json_decode($response['body']));

            throw new Exception('API Error: Unable to despatch order ' . $order_id . ' [' . $msg . ']');
        }
    }


    public function SaveProduct($product)
    {
        // All submissions must be in json
        $json = json_encode($product);

        if (is_numeric($product['product']['id'])) {
            // PUT request for product updates
            $api = [
                'url' => 'products/' . $product['product']['id'] . '.json',
                'config' => [
                    CURLOPT_CUSTOMREQUEST => 'PUT',
                    CURLOPT_POSTFIELDS => $json,
                    CURLOPT_HTTPHEADER => [
                        'Content-Type: application/json',
                        'Content-Length: ' . strlen($json),
                    ],
                ],
            ];

            // Make the api request
            $response = $this->_ApiCall($api);
            if ($response['http_code'] != 200) {
                $msg = $this->_GetErrorMessage(json_decode($response['body']));
                throw new Exception('API Error: Unable to update product "' . $product['product']['title'] . '" [' . $msg ? $msg : $response . ']');
            }
        } else {
            // POST request for new product
            $api = [
                'url' => 'products.json',
                'config' => [
                    CURLOPT_POST => true,
                    CURLOPT_POSTFIELDS => $json,
                    CURLOPT_HTTPHEADER => [
                        'Content-Type: application/json',
                        'Content-Length: ' . strlen($json),
                    ],
                ],
            ];

            // Make the api request
            $response = $this->_ApiCall($api);
            if ($response['http_code'] != 201) {
                $msg = $this->_GetErrorMessage(json_decode($response['body']));
                throw new Exception('API Error: Unable to add product "' . $product['product']['title'] . '" [' . $msg ? $msg : $response . ']');
            }
        }
    }


    public function SaveVariant($variant, $sku = '')
    {
        // All submissions must be in json
        $json_data = json_encode($variant);

        if (is_numeric($variant['variant']['id'])) {
            // PUT request for variant update
            $api = [
                'url' => 'variants/' . $variant['variant']['id'] . '.json',
                'config' => [
                    CURLOPT_HEADER => true,
                    CURLOPT_CUSTOMREQUEST => 'PUT',
                    CURLOPT_POSTFIELDS => $json_data,
                    CURLOPT_HTTPHEADER => [
                        'Content-Type: application/json',
                        'Content-Length: ' . strlen($json_data),
                    ],
                ],
            ];

            // Make the api request
            $response = $this->_ApiCall($api);

            if ($response['http_code'] != 200) {
                $msg = $this->_GetErrorMessage(json_decode($response['body']));
                throw new Exception('API Error: Unable to update variant "' . $sku . '" [' . $msg ? $msg : $response . ']');
            }
        } else {
            if (is_numeric($variant['variant']['product_id'])) {
                // POST request for new variant
                $api = [
                    'url' => 'products/' . $variant['variant']['product_id'] . '/variants.json',
                    'config' => [
                        CURLOPT_POST => true,
                        CURLOPT_POSTFIELDS => $json_data,
                        CURLOPT_HTTPHEADER => [
                            'Content-Type: application/json',
                            'Content-Length: ' . strlen($json_data),
                        ],
                    ],
                ];

                // Make the api request
                $response = $this->_ApiCall($api);

                if ($response['http_code'] != 201) {
                    $msg = $this->_GetErrorMessage(json_decode($response['body']));
                    throw new Exception('API Error: Unable to add/update variant "' . $sku . '" [' . $msg ? $msg : $response . ']');
                }
            } else {
                throw new Exception('Error: Product or variant id missing');
            }
        }
    }

    
    /**
     * 22/05/2019 | TL | Dev 43368 | Script Edit to handle multiple stock locations | function to call and return stock location information.
     * Returns the current stock levels of Shopify variants
     *
     * @return array $inventory
     *  Load the inventory as an indexed array; to allow duplicate skus
     */
    public function LoadCurrentStockLevels()
    {
        $endpoint = 'products.json?limit=' . self::PRODUCT_PAGINATION_LIMIT . '&fields=id,variants';
        $inventory = [];

        do {
            // 17/02/2020 | CB | Dev 47605 | API call will add pagination as required
            $response = $this->_ApiCall(['url' => $endpoint], 'stock');
            if (empty($response)) {
                return $inventory;
            }

            $r = json_decode($response['body']);
            if (!empty($r->products)) {
                foreach ($r->products as $product) {

                    // All products have at least one variant which is created when the product is initially created
                    foreach ($product->variants as $variant) {
                        // only add the variant if it has its inventory managed by Shopify
                        if ($variant->sku != "" && $variant->inventory_management == 'shopify') {
                            $inventory[strtoupper($variant->sku)][] = [
                                'variant_id' => $variant->id,
                                'qty' => $variant->inventory_quantity,
                                'inventory_item_id' => $variant->inventory_item_id,
                            ];
                        }
                    }
                }
            }
        } while ($this->isNextPageAvailable('stock'));

        return $inventory;
    }


    /**
     * Returns the current pricing of shopify variants
     *
     * @return array $inventory
     */
    public function LoadCurrentPricing()
    {
        $endpoint = 'products.json?limit=' . self::PRODUCT_PAGINATION_LIMIT . '&fields=id,variants';
        $inventory = [];

        do {
            // 17/02/2020 | CB | Dev 47605 | API call will add pagination as required
            $response = $this->_ApiCall(['url' => $endpoint], 'price');
            if (empty($response)) {
                return $inventory;
            }

            $r = json_decode($response['body']);
            if (count($r->products) > 0) {
                foreach ($r->products as $product) {
                    foreach ($product->variants as $variant) {
                        if ($variant->sku != "") {
                            // for consistency when comparing SKUs we do everything in uppercase
                            $sku = strtoupper($variant->sku);
                            // 24/07/2019 | DS | Dev 44582 | Amendment to update all prices that share the same SKU by inserting variants into an array instead of overwriting the last one with the SKU
                            $inventory[$sku][] = ['variant_id' => $variant->id, 'price' => $variant->price];
                        }
                    }
                }
            }
        } while ($this->isNextPageAvailable('price'));

        return $inventory;
    }


    /**
     * Get the Shopify stock location ID where only one stock location is setup in Shopify
     *
     * @return string
     * @throws Exception
     */
    public function GetSingleStockLocationIDOnly()
    {
        // Make the request
        $response = $this->_ApiCall(['url' => 'locations.json']);

        // Decode the response
        $r = json_decode($response['body']);
        if (isset($r->errors)) {
            $msg = $this->_GetErrorMessage(json_decode($response['body']));
            throw new Exception('API Error: Unable to retrieve stock locations ' . $msg ? $msg : $response);
        }

        if (count($r->locations) != 1) {
            throw new Exception('Multiple stock locations setup in Shopify - please contact OrderWise Support');
        }

        $stock_location_id = trim($r->locations[0]->id);

        if (!$stock_location_id) {
            throw new Exception('Could not get stock location ID to use');
        }

        return $stock_location_id;
    }


    //01/07/2019 | DS | Dev 43934 | Get the already-loaded stock locations without doing another API call
    public function IsMultipleStockLocation()
    {
        return count($this->stock_locations) > 1;
    }

    //22/05/2019 | TL | Dev 43368 | Script Edit to handle multiple stock locations | Get Shopify Stock Location ID based of the name.
    // Used within the Despatch and Stock Scripts.

    public function GetFirstStockLocation()
    {
        foreach ($this->stock_locations as $locations) {
            $stock_location_id = $locations->id;

            return $stock_location_id;
        }
        throw new Exception('No stock locations found in Shopify. Please check your store configuration');
    }


    public function GetLoadedStockLocations()
    {
        return $this->stock_locations;
    }


    public function GetLocationIDFromName($location_name)
    {
        if (!empty($location_name)) {
            foreach ($this->stock_locations as $locations) {
                if (strtoupper($locations->name) == strtoupper($location_name)) {
                    $stock_location_id = $locations->id;

                    return $stock_location_id;
                }
            }
        }
        throw new Exception('Invalid Stock Location Name. Please ensure the OrderWise stock location name matches a Shopify stock location name, or map the stock locations via the stock_locations.csv file if the stock location names cannot be changed to match');
    }


    /**
     * @param $sku
     * @param $update
     * @return bool
     * @throws Exception
     */
    public function UpdateStockLevelForSpecificLocation($sku, $update)
    {
        $base_message = "Unable to update variant {$sku}: ";

        if (!is_numeric($update['inventory_item_id'])) {
            throw new Exception($base_message . "Variant id missing");
        }

        if (!is_numeric($update['location_id'])) {
            throw new Exception($base_message . "Stock location id missing");
        }

        $json_data = json_encode($update);
        $api = [
            'url' => 'inventory_levels/set.json',
            'config' => [
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => $json_data,
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/json',
                    'Content-Length: ' . strlen($json_data),
                ],
            ],
        ];

        // Make the api request
        $response = $this->_ApiCall($api);
        if ($response['http_code'] != 200) {
            $msg = $this->_GetErrorMessage(json_decode($response['body']));
            throw new Exception('API Error: Unable to update variant "' . $sku . '" [' . $msg ? $msg : $response . ']');
        }

        return true;
    }


    // 10/08/2018 | CB | Dev 38232 | Check if passed title is a tax title, based on set list from the client
    public function isTaxTitle($title)
    {
        if (in_array($title, $this->_tax_titles)) {
            return true;
        }

        return false;
    }


    // 17/10/2018 | AC | Dev ID 39000 | Retrieve all tax codes from the Shopify countries API
    public function getTaxTitles()
    {
        $response = $this->_ApiCall(['url' => 'countries.json']);

        // Decode the response
        $r = json_decode($response['body']);
        if (isset($r->errors)) {
            $msg = $this->_GetErrorMessage(json_decode($response['body']));
            throw new Exception('API Error: Unable to retrieve order country tax codes [' . $msg ? $msg : $response . ']');
        }

        //Loop through all countries and provinces to pull all tax codes
        if (count($r->countries) > 0) {
            foreach ($r->countries as $country) {
                if (!is_null($country->tax_name)) {
                    $this->_tax_titles[$country->tax_name] = $country->tax_name;
                }
                if (count($country->provinces) > 0) {
                    foreach ($country->provinces as $province) {
                        if (!is_null($province->tax_name)) {
                            $this->_tax_titles[$province->tax_name] = $province->tax_name;
                        }
                    }
                }
            }
        }
    }


    /**
     *
     * Dev 47605
     *
     * New methods added for cursor pagination
     * https://help.shopify.com/en/api/guides/paginated-rest-results
     * This is forced in 2019-07 onwards so old way of pagination removed
     */

    /**
     * Set the previous and next cursor pagination links based on response from API
     *
     * @param $cursor_type
     * @param $response
     * @throws Exception
     */
    private function setCursorPaginationLinks($cursor_type, $response)
    {
        if (empty($this->_current_cursor_pagination_urls[$cursor_type])) {
            throw new Exception('Invalid cursor type to when obtaining cursor pagination links');
        }

        if (empty($response['content'])) {
            throw new Exception('Could not load response content to get cursor pagination links');
        }

        $response_content = $response['content'];
        if (strpos($response_content, ',') !== false) {
            $headers = explode(',', $response_content);
        } else {
            $headers[] = $response_content;
        }

        $pagination_links = [];
        foreach ($headers as $response_content) {

            $header_rel = $this->extractRelFromHeader($response_content);
            $header_rel_data = $this->extractPageInfoFromHeader($response_content);

            if (!empty($header_rel) && !empty($header_rel_data)) {
                $pagination_links += [$header_rel => $header_rel_data];
            }
        }

        // Empty if no pages or just one page available
        if (empty($pagination_links)) {
            $this->_current_cursor_pagination_urls[$cursor_type]['is_next_page_available'] = false;

            return true;
        }

        $old_next_cursor = $this->_current_cursor_pagination_urls[$cursor_type]['next'];
        $this->_current_cursor_pagination_urls[$cursor_type] = array_filter($pagination_links);
        $new_next_cursor = $this->_current_cursor_pagination_urls[$cursor_type]['next'];

        // Check just in case we somehow get caught in a loop
        if (!empty($new_next_cursor) && $old_next_cursor == $new_next_cursor) {
            throw new Exception('Pagination looping - could not continue');
        }

        $this->_current_cursor_pagination_urls[$cursor_type]['is_next_page_available'] = !empty($this->_current_cursor_pagination_urls[$cursor_type]['next']);

        return true;
    }


    /**
     * Extract the href from page header
     * https://help.shopify.com/en/api/guides/paginated-rest-results
     *
     * @param $element
     * @return mixed
     */
    private function extractPageInfoFromHeader($element)
    {
        if (preg_match('/page_info=(.*\w)[>|&|;]/', $element, $match) == 1) {
            return $match[1];
        }
    }


    /**
     * Get the API version from the headers given back from Shopify
     * Format of header =  X-Shopify-API-Version: 2019-04
     *
     * @param $response_content
     */
    private function extractApiVersionFromHeader($response_content)
    {
        if (preg_match('/X-Shopify-API-Version: (.*)/', $response_content, $match) == 1) {
            return trim($match[1]);
        }

        return false;
    }


    /**
     * Extract the 'rel' from header
     * https://help.shopify.com/en/api/guides/paginated-rest-results
     *
     * @param $element
     * @return mixed
     */
    private function extractRelFromHeader($element)
    {
        if (preg_match('/rel="(.*?)"/', $element, $match) == 1) {
            return $match[1];
        }
    }


    /**
     * Get the next page URL based on the cursor pagination given in the last API call
     *
     * @param $cursor_type
     * @param $base_url
     * @return string
     * @throws Exception
     */
    public function getNextPageUrl($cursor_type, $base_url)
    {
        if (empty($this->_current_cursor_pagination_urls[$cursor_type])) {
            throw new Exception('Could not find type to calculate the next page url');
        }

        if (!$this->isNextPageAvailable($cursor_type))
        {
            return false;
        }

        $current_pagination = $this->_current_cursor_pagination_urls[$cursor_type];

        if (empty($current_pagination['previous']) && empty($current_pagination['next'])) {
            return $base_url;
        }

        if (!empty($current_pagination['next'])) {
            return $base_url . '&page_info=' . $current_pagination['next'];
        }

        return $base_url;
    }


    /**
     * Check if there is a next page URL available
     *
     * @param $cursor_type
     * @return bool
     * @throws Exception
     */
    private function isNextPageAvailable($cursor_type)
    {
        if (empty($this->_current_cursor_pagination_urls[$cursor_type])) {
            throw new Exception('Invalid cursor type when finding if next page is available');
        }

        if (empty($this->_current_cursor_pagination_urls[$cursor_type]['is_next_page_available'])) {
            return false;
        }

        return $this->_current_cursor_pagination_urls[$cursor_type]['is_next_page_available'];
    }


    private function isFirstPageCall($cursor_type)
    {
        if (empty($this->_current_cursor_pagination_urls[$cursor_type])) {
            throw new Exception('Invalid cursor type when finding if this is the call for the first page');
        }

        $current_pagination = $this->_current_cursor_pagination_urls[$cursor_type];

        return ($current_pagination['is_next_page_available'] == true
                    && empty($current_pagination['previous'])
                    && empty($current_pagination['next'])
        );
    }


    // 13/07/2020 | CB | Dev 49316 | Added
    public function LoadCurrentBarcodes()
    {
        $endpoint = 'products.json?limit=' . self::PRODUCT_PAGINATION_LIMIT . '&fields=id,variants';
        $inventory = [];

        do {
            $response = $this->_ApiCall(['url' => $endpoint], 'barcodes');
            if (empty($response)) {
                return $inventory;
            }

            $r = json_decode($response['body']);
            if (count($r->products) > 0) {
                foreach ($r->products as $product) {
                    foreach ($product->variants as $variant) {
                        if ($variant->sku != "") {
                            // for consistency when comparing SKUs we do everything in uppercase
                            $sku = strtoupper($variant->sku);
                            $inventory[$sku][] = ['variant_id' => $variant->id, 'barcode' => $variant->barcode];
                        }
                    }
                }
            }
        } while ($this->isNextPageAvailable('barcodes'));

        return $inventory;
    }
}