<?php

class Settings
{
    private $configuration_errors;
    private $settings_errors;
    private $settings_xml;
    private $script_options;
    private $core_options;
    // XML set to indicate what is being saved (the core config or script-specific config)
    private $save_xml;

    /**
     * Settings constructor.
     * @param bool $save - used to indicate if the settings should be saved after loading the XML (done on page load after posting data to update the settings)
     */
    function __construct($save = false)
    {
        $this->configuration_errors = [];
        $this->settings_errors = [];

        // options specific to each script implementation
        $this->script_options = [];

        // core options that consist as part of the base scripts (not script specific)
        $this->core_options = [];

        $this->settings_xml = new stdClass();
        $this->save_xml = new stdClass();

        if (file_exists("settings.xml")) {
            $this->settings_xml = simplexml_load_file("settings.xml");
            if ($save) {
                $this->Save();
            }
        }
        $this->LoadSettings();
    }

    private function Save()
    {
        // update the XML file with the posted settings
        if (!empty($_GET['save']) && !empty($_POST)) {
            $new_settings = $_POST;
            switch ($_GET['save']) {
                case 'core':
                    $this->save_xml = &$this->settings_xml->core;
                    $this->SaveSettings($new_settings);
                    $this->configuration_errors[] = [
                        'type' => 'success',
                        'message' => 'Saved core settings'
                    ];
                    break;
                case 'script':
                    $this->save_xml = &$this->settings_xml->script;
                    $this->SaveSettings($new_settings);
                    $this->settings_errors[] = [
                        'type' => 'success',
                        'message' => 'Saved script settings'
                    ];
                    break;
                default:
                    // should never reach this
                    $this->configuration_errors[] = [
                        'type' => 'error',
                        'message' => 'Invalid save option specified'
                    ];
                    break;
            }
        }
    }

    private function SaveSettings($settings)
    {
        foreach ($settings as $array_key => $value) {
            $option_id = array_search($array_key, array_keys($settings));
            if (isset($this->save_xml->option[$option_id]->value) && (isset($this->save_xml->option[$option_id]->type))) {
                switch (strtolower($this->save_xml->option[$option_id]->type)) {
                    case 'checkbox':
                        $this->save_xml->option[$option_id]->value = !empty($value) ? 'true' : 'false';
                        break;
                    case 'text':
                    case 'select':
                    case 'number':
                    case 'date':
                        $this->save_xml->option[$option_id]->value = $value;
                        break;
                    default:
                        $this->settings_errors[] = [
                            'type' => 'error',
                            'message' => 'attempted to update an unknown option'
                        ];
                        break;
                }

            }
        }
        $this->settings_xml->asXML('settings.xml');
    }

    private function LoadSettings()
    {
        // load settings from the settings.xml
        if ($this->settings_xml) {
            if (!empty($this->settings_xml)) {
                // base script configuration options
                if (!empty($this->settings_xml->core) && !empty($this->settings_xml->core->option)) {
                    foreach ($this->settings_xml->core->option as $option) {
                        $this->LoadOption($this->core_options, $option);
                    }
                } else {
                    $configuration_errors[] = [
                        'type' => 'error',
                        'message' => "No core options configured in the options.xml"
                    ];
                }

                // script specific options
                if (!empty($this->settings_xml->script) && !empty($this->settings_xml->script->option)) {
                    foreach ($this->settings_xml->script->option as $option) {
                        $this->LoadOption($this->script_options, $option);
                    }
                }
            } else {
                $configuration_errors[] = [
                    'type' => 'error',
                    'message' => 'Options XML file is empty',
                ];
            }
        } else {
            $configuration_errors[] = [
                'type' => 'error',
                'message' => 'unable to load the options XML file',
            ];
        }
    }

    /**
     * Generic option function to load an option from the XML
     * @param $options_array
     * @param $option
     */
    private function LoadOption(&$options_array, $option)
    {
        if (empty($option->name) || empty($option->description) || empty($option->label) || empty($option->type)) {
            $configuration_errors[] = [
                'type' => 'warning',
                'message' => "Option not configured correctly"
            ];
        } else {
            $options_array[trim($option->name)] = array(
                'label' => trim($option->label),
                'name' => trim($option->name),
                'description' => trim($option->description),
                'value' => trim($option->value),
                'type' => trim($option->type),
            );

            if (!empty($option->suboptions)) {
                $suboptions = array();
                foreach ($option->suboptions->suboption as $suboption) {
                    $suboptions[trim($suboption->value)] = array(
                        'label' => !empty($suboption->label) ? trim($suboption->label) : '',
                        'enabled' => !empty($suboption->enabled) && strtolower($suboption->enabled) == 'true',
                        'value' => !empty($suboption->value) ? trim($suboption->value) : '',
                    );
                }
                $options_array[trim($option->name)]['suboptions'] = $suboptions;
            }
        }
    }

    /**
     * @param $type
     * @param $message
     */
    public function AddSettingError($type, $message)
    {
        $this->settings_errors[] = [
            'type' => strtolower($type),
            'message' => $message
        ];
    }

    /**
     * @param $type
     * @param $message
     */
    public function AddConfigurationError($type, $message)
    {
        $this->configuration_errors[] = [
            'type' => $type,
            'message' => $message
        ];
    }

    /**
     * @param $option_name
     * @param $suboption_name
     */
    public function DisableCoreSuboption($option_name, $suboption_name)
    {
        // if the requested core option is found, disable it
        if (!empty($this->core_options[$option_name]['suboptions'][$suboption_name]['enabled']))
            $this->core_options[$option_name]['suboptions'][$suboption_name]['enabled'] = false;
    }

    /**
     * @param $option_name
     * @return bool|mixed|null
     */
    public function GetScriptOption($option_name)
    {
        $option_name = strtolower($option_name);
        if (!empty($this->script_options[$option_name]) && !empty($this->script_options[$option_name]['value'])) {
            if ($this->script_options[$option_name]['type'] == 'checkbox') {
                return (strtolower($this->script_options[$option_name]['value']) == 'true');
            } else {
                return ($this->script_options[$option_name]['value']);
            }
        }
        return null;
    }

    /**
     * @param $option_name
     * @return array
     */
    public function GetCoreOption($option_name)
    {
        $option_name = strtolower($option_name);
        return (!empty($this->core_options[$option_name]) && !empty($this->core_options[$option_name]['value']) ? $this->core_options[$option_name]['value'] : null);
    }

    /**
     * @return array
     */
    public function GetConfigurationErrors()
    {
        return $this->configuration_errors;
    }

    /**
     * @return array
     */
    public function GetScriptErrors()
    {
        return $this->settings_errors;
    }

    /**
     * @return array
     */
    public function GetCoreOptions()
    {
        return $this->core_options;
    }

    /**
     * @return array
     */
    public function GetScriptOptions()
    {
        return $this->script_options;
    }
	
	 /**
     * @param $order_statuses_string
     * @return string
     */
    public function SanitiseOrderStatuses($order_statuses_string)
    {
        $order_statuses_string = filter_var($order_statuses_string, FILTER_SANITIZE_STRING);

        // Convert to array and back to remove errant spaces
        $order_statuses_array = array_map('trim', explode(',', $order_statuses_string));

        return implode(',', $order_statuses_array);
    }
	
	public function SanitiseAnalysisField($value = null)
    {
        if (empty($value)){
            return null;
        }

        return strtoupper(filter_var($value, FILTER_SANITIZE_STRING));
    }
}