<?php

class Handler
{
    const OWS_IMPORTED_ORDER_TABLE_NAME = '#__ows_imported_order';
    public $_allow_matching_delivery_contact_name = false;
    public $_use_company_name_as_statement_name = true;
    protected $_db;
    private $_mbstring_available;
    private $_xml_response_messages = []; // setting to allow delivery and contact fields to match, required for some carriers
    private $_config; // Setting to control company and billing name mapping.
    private $_order_counter = 0;
    private $_xml;

    function __construct()
    {
        $this->_config = new stdClass();
        $this->_config->directories = [];
        $this->SetOrderAgeDaysBack(7);

        if (extension_loaded('mbstring')) {
            mb_internal_encoding('UTF-8');
            $this->_mbstring_available = true;
        } else {
            $this->_mbstring_available = false;
        }
    }

    /**
     * @param $v
     */
    public function SetOrderAgeDaysBack($v)
    {
        $this->_config->order_age_days_back = $v;
    }

    /**
     * Loads the script configuration controlled by setup.php
     * @param Settings $settings
     */
    public function LoadConfiguration($settings)
    {
        // If there isn't a settings.xml file we cannot continue
        if (!file_exists('settings.xml')) {
            $this->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, 'No settings.xml file found - please use the cart-specific settings.xml file.');
            $this->ReturnResponseXML();
        }

        // Load the settings from the settings manager via the setting name (settings can be added in the settings.xml and their toggled in settings.php)
        $db_mode = $settings->GetCoreOption('web_database_connection_target');
        $maximum_orders_per_session = $settings->GetCoreOption('maximum_orders_per_session');
        $order_age_days_back = $settings->GetCoreOption('order_age_days_back');
        $order_after_date = $settings->GetCoreOption('order_after_date');
        $sku_filter = $settings->GetCoreOption('sku_filter');

        //set the configuration settings
        $this->SetDbMode($db_mode);
        $this->SetMaximumOrdersPerSession($maximum_orders_per_session);
        if (is_numeric($order_age_days_back)) {
            $this->SetOrderAgeDaysBack($order_age_days_back);
        }
        $this->SetOrderAfterDate($order_after_date);
        $this->SetSkuFilter($sku_filter);
    }

    /**
     * Adds a message to the response array which is added to any xml
     * returned to OrderWise
     *
     * @param string $type
     * @param string $msg
     */
    public function AppendResponseMessage($type, $msg)
    {
        $this->_xml_response_messages[] = ['type' => $type, 'message' => $msg];
    }

    /**
     * Returns standalone response XML when system errors are generated
     * These include such things as not connecting to a database
     */
    public function ReturnResponseXML()
    {
        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (empty($this->_xml_response_messages)) {
            exit;
        }
        header('Cache-Control: no-cache');
        header('Pragma: no-cache');
        header('content-type: text/xml; charset=utf-8');
        ?>
        <XMLFile>
            <Responses>
                <?php
                foreach ($this->_xml_response_messages as $k => $v) {
                    // All response messages are written to the script logs for reference
                    WriteToLog($v['message']);
                    ?>
                    <Response>
                        <Type><?php echo $v['type']; ?></Type>
                        <Message>[WebScript] <?php echo htmlspecialchars($v['message'], ENT_QUOTES); ?></Message>
                    </Response>
                    <?php
                }
                ?>
            </Responses>
        </XMLFile>
        <?php
        exit;
    }

    /**
     * @param $v
     */
    public function SetDbMode($v)
    {
        $this->_config->db_mode = $v;
    }

    /**
     * @param $v
     */
    public function SetMaximumOrdersPerSession($v)
    {
        $this->_config->maximum_orders_per_session = $v;
    }

    /**
     * @param $v
     */
    public function SetOrderAfterDate($v)
    {
        $this->_config->order_after_date = $v;
    }

    /**
     * @param $v
     */
    public function SetSkuFilter($v)
    {
        $this->_config->sku_filter_list = [];
        $this->_config->sku_filter_cnt = 0;
        if ($v != '') {
            $list = explode(',', strtoupper($v));
            foreach ($list as $sku) {
                $sku = trim($sku);
                $this->_config->sku_filter_list[$sku] = $sku;
            }
            $this->_config->sku_filter_cnt = !empty($this->_config->sku_filter_list) ? count($this->_config->sku_filter_list) : 0;
        }
    }

    /**
     * Creates the script directory structure and deletes aged files
     */
    public function DirectoryManagement()
    {
        foreach ($this->GetDirectory() as $dir_name => $d) {
            $dirs = explode('/', $d);
            $path = '';
            foreach ($dirs as $dir) {
                if ($dir != '') {
                    if ($path != '') {
                        $path .= '/';
                    }
                    $path .= $dir;
                    $this->_CreateDirectory($path);
                    $this->_RemoveOldFiles($path);
                }
            }
        }
        WriteToLog('Directory management complete');
    }

    /**
     * @param string $n
     * @return mixed
     */
    public function GetDirectory($n = '')
    {
        if ($n != '') {
            return $this->_config->directories[$n];
        }
        return $this->_config->directories;
    }

    /**
     * Creates a new directory
     *
     * @param string $dir
     */
    private function _CreateDirectory($dir)
    {
        if ($dir == '') {
            return;
        }
        if (is_dir($dir)) {
            return;
        }
        if (mkdir($dir, 0755)) {
            return;
        }
        $this->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, 'Script terminated: unable to create \'' . $dir . '\' directory.');
        $this->ReturnResponseXML();
    }

    /**
     * Removes files from a directory over a number of days old
     * this defaults to 7 days and is controlled by _file_removal_days_back
     *
     * @param string $dir
     */
    private function _RemoveOldFiles($dir)
    {
        // Remove files over $this->_file_removal_days_back days old
        if ($dir != '') {
            if ($dh = @opendir($dir)) {
                while (($fn = @readdir($dh)) !== false) {
                    if (is_dir($dir . '/' . $fn)) {
                        continue;
                    }
                    $access_date = date('Ymd', fileatime($dir . '/' . $fn));
                    if ($fn != '.' && $fn != '..' && $fn != '.htaccess' && $access_date < date('Ymd', strtotime('-7 days'))) {
                        @unlink($dir . '/' . $fn);
                    }
                }
            }
        }
    }

    /**
     * Creates a new order import object and should be called
     * when creating each order to be added to the import
     *
     * @return object
     */
    public function NewOrder()
    {
        $d = new StdClass();
        $d->order_lines = [];
        $d->payments = [];
        $d->discounts = [];
        $d->analysis['sales_order'] = [];
        $d->analysis['customer'] = [];
        $d->analysis['contact'] = [];
        $d->analysis['delivery'] = [];
        $d->customer['statement_addr'] = new stdClass();
        $d->customer['invoice_addr'] = new stdClass();
        $d->customer['delivery_addr'] = new stdClass();
        $d->customer['contact'] = new stdClass();
        return $d;
    }

    /**
     * Creates a new customer import object and should be called
     * when creating each customer to be added to the import
     *
     * @return object
     */
    public function NewCustomer()
    {
        $d = new StdClass();
        $d->customer['statement_addr'] = new stdClass();
        $d->customer['delivery_addr'] = new stdClass();
        $d->customer['contact'] = new stdClass();
        return $d;
    }

    /**
     * Creates an archive of data posted to a script. Used in stock, price
     * and despatch sync to store the OrderWise exported data.
     *
     * @param string $fn
     * @param string $data
     * @return bool
     */
    public function ArchiveOrderwiseExportedData($fn, $data)
    {
        // XML extension is hard coded as we always work with XML exports from orderwise
        if (is_dir($this->GetDirectory('exports_archive'))) {
            $fn = $this->GetDirectory('exports_archive') . $fn . '.' . $this->GetStoreId() . '.' . date('YmdHis') . '.xml';
            if (file_put_contents($fn, $data) !== false) {
                return true;
            }
        }
        $this->AppendResponseMessage(XML_REPONSE_TYPE_WARNING, 'Unable to archive export data');
    }

    /**
     * @return mixed
     */
    public function GetStoreId()
    {
        return $this->_config->store_id;
    }

    /**
     * Orders imported into OrderWise are posted to the orders.php script and
     * these are recorded in the database to be excluded from the next import.
     *
     * @param array $post_data
     */
    public function MarkOrdersImported($post_data)
    {
        $post_data = stripslashes($post_data['ExportData']);
        if (!empty($post_data)) {
            $arr = explode('~', $post_data);
            foreach ($arr as $order_id) {
                if ($order_id != '') {
                    $sql = $this->_db->Query('	INSERT INTO ' . self::OWS_IMPORTED_ORDER_TABLE_NAME . ' (
													order_id,
													store_id,
													date_imported
												) VALUES (
													"' . $this->_db->FixUserQuery($order_id) . '",
													"' . $this->GetStoreId() . '",
													NOW()
												)');
                }
            }
            WriteToLog(count($arr) . ' flagged as imported');
            return;
        }
        WriteToLog('No valid post data found.');
    }

    /**
     * When converting files for import this removes a successfully imported
     * file from the 'downloads_pending' directory
     *
     * @param string $post_data
     */
    public function MarkOrderFilesImported($post_data)
    {
        $order_ids = [];
        $posted_data = stripslashes($post_data['ExportData']);
        if (!empty($posted_data)) {
            $arr = explode('~', $posted_data);
            foreach ($arr as $order_no) {
                if ($order_no != '') {
                    $order_ids[] = $order_no;
                }
            }
        }

        if (count($order_ids) == 0) {
            WriteToLog('No valid post data found');
            return;
        }

        $format = new DataFormat();
        WriteToLog(count($order_ids) . ' exported order ids being processed');
        foreach ($order_ids as $order_id) {
            $order_id_filename = $format->StrToSafeChars($order_id) . '.xml';
            $order_file = $this->GetDirectory('downloads_pending') . $order_id_filename;
            if (file_exists($order_file)) {
                if (!rename($order_file, $this->GetDirectory('downloads_archive') . $order_id_filename)) {
                    WriteToLog('unable to move file ' . $order_id . ' to archive directory');
                }
            }
        }
    }

    /**
     * Returns a list or previously import order ids. This won't be all
     * imported ids, just those that fall within the the order download date range
     *
     * @return array
     */
    public function GetPreviouslyImportedOrderIds()
    {
        $order_ids = [];
        $sql = $this->_db->Query('	SELECT
										order_id
									FROM
										' . self::OWS_IMPORTED_ORDER_TABLE_NAME . '
									WHERE
										store_id = "' . $this->GetStoreId() . '"
										AND date_imported >= "' . $this->GetOrdersFromDate() . '"');
        while ($row = $this->_db->FetchObject($sql)) {
            if ($row->order_id != '') {
                $order_ids[$row->order_id] = true;
            }
        }

        WriteToLog((count($order_ids)) . ' previously imported order ids found');
        return $order_ids;
    }

    /**
     * Calculates the orders from date to be incorporated into the orders
     * query if required. If no order_age_days_back or order_after_date
     * values are set, this will default to calculating 7 days back
     *
     * @return bool|string
     */
    public function GetOrdersFromDate()
    {
        if ($this->GetOrderAgeDaysBack() == 0 && $this->GetOrderAfterDate() == 0) {
            $this->AppendResponseMessage(XML_REPONSE_TYPE_WARNING, "Either 'order_age_days_back' or 'order_after_date' needs to be set in config.inc.php");
            $this->ReturnResponseXML();
        }
        $d1 = date('Y-m-d', strtotime(($this->GetOrderAgeDaysBack()) . ' days ago'));
        $d2 = date('Y-m-d', strtotime($this->GetOrderAfterDate() . ' +1 day'));
        if ($this->GetOrderAgeDaysBack() == 0) {
            $d = $d2;
        } else {
            if ($this->GetOrderAfterDate() == 0) {
                $d = $d1;
            } else {
                $d = ($d2 > $d1) ? $d2 : $d1;
            }
        }

        // When the date is set, the select orders query should use 'order_date > $d'
        return $d;
    }

    /**
     * @return mixed
     */
    public function GetOrderAgeDaysBack()
    {
        return $this->_config->order_age_days_back;
    }

    /**
     * @return mixed
     */
    public function GetOrderAfterDate()
    {
        return $this->_config->order_after_date;
    }

    /**
     * Whether an order id exists in the previously downloaded table
     *
     * @return bool
     */
    public function IsPreviouslyImportedOrder($order_id)
    {
        $cnt = $this->_db->Result($this->_db->Query('SELECT COUNT(*) AS total FROM ' . self::OWS_IMPORTED_ORDER_TABLE_NAME . ' WHERE order_id = "' . $this->_db->FixUserQuery($order_id) . '" AND store_id = ' . $this->GetStoreId()));
        if ($cnt == 0) {
            return false;
        }
        return true;
    }

    /**
     * Creates the XML headers for returning customer xml to OrderWise
     */
    public function CreateCustomerXmlHeaders()
    {
        // Start the xml document and create the root element based on the ow version
        $this->_xml = new stdClass();
        $this->_xml->dom = new domDocument('1.0', 'UTF-8');
        $this->_xml->root = $this->_xml->dom->createElement('XMLFile');
        $this->_xml->root->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:xsd', 'http://www.w3.org/2001/XMLSchema');
        $this->_xml->root->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
        $this->_xml->root = $this->_xml->dom->appendChild($this->_xml->root);

        $this->_xml->reponses = $this->_xml->dom->createElement('Responses');
        $this->_xml->root->appendChild($this->_xml->reponses);

        $this->_xml->customers = $this->_xml->dom->createElement('Customers');
        $this->_xml->root->appendChild($this->_xml->customers);
    }

    /**
     * Creates the XML headers for returning order xml to OrderWise
     */
    public function CreateOrderXmlHeaders()
    {
        // Start the xml document and create the root element based on the ow version
        $this->_xml = new stdClass();
        $this->_xml->dom = new domDocument('1.0', 'UTF-8');
        $this->_xml->root = $this->_xml->dom->createElement('XMLFile');
        $this->_xml->root->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:xsd', 'http://www.w3.org/2001/XMLSchema');
        $this->_xml->root->setAttributeNS('http://www.w3.org/2000/xmlns/', 'xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
        $this->_xml->root = $this->_xml->dom->appendChild($this->_xml->root);

        $this->_xml->reponses = $this->_xml->dom->createElement('Responses');
        $this->_xml->root->appendChild($this->_xml->reponses);

        $this->_xml->orders = $this->_xml->dom->createElement('SalesOrders');
        $this->_xml->root->appendChild($this->_xml->orders);
    }

    /**
     * Builds the v7 XML order import data
     *
     * @param $order_details
     * @return bool
     */
    public function AppendOrderToXml($order_details)
    {
        // Create the order element
        $order = $this->_xml->dom->createElement('SalesOrder');
        $this->_xml->orders->appendChild($order);

        // Add custom attributes
        $attributes = $this->_xml->dom->createElement('SalesOrderAttributes');
        $order->appendChild($attributes);

        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($order_details->sales_order_attributes)) {
            foreach ($order_details->sales_order_attributes as $attribute_key => $attribute_value) {
                $attribute_node = $this->_xml->dom->createElement('SalesOrderAttribute');
                $attributes->appendChild($attribute_node);
                $this->_AddElement($attribute_node, 'AttributeName', $attribute_key);
                $this->_AddElement($attribute_node, 'AttributeValue', $attribute_value);
            }
        }

        // Add the order details to the order element
        $this->_AddElement($order, 'OrderNumber', $order_details->order_id);
        $this->_AddElement($order, 'OrderAnalysis', $order_details->order_analysis);
        $this->_AddElement($order, 'SpecialInstructions', $order_details->order_special_instructions);
        $this->_AddElement($order, 'SpecialDeliveryInstructions', $order_details->delivery_special_instructions);
        $this->_AddElement($order, 'CustomerOrderRef', $order_details->customer_order_ref);

        if (!empty($order_details->order_on_hold) && $order_details->order_on_hold) {
            $this->_AddElement($order, 'OrderOnHold', 'true');
        }

        // Set the various order related dates
        if (strtotime($order_details->order_date) != null) {
            $this->_AddElement($order, 'OrderDate', date('Y-m-d\TH:i:s', strtotime($order_details->order_date)));
        }
        if (strtotime($order_details->order_required_date) != null) {
            $this->_AddElement($order, 'RequiredDate', date('Y-m-d\TH:i:s', strtotime($order_details->order_required_date)));
        }
        if (strtotime($order_details->order_promised_date) != null) {
            $this->_AddElement($order, 'PromisedDate', date('Y-m-d\TH:i:s', strtotime($order_details->order_promised_date)));
        }

        // 13/03/2020 | CB | Dev 48957 | Added system calc. nodes
        if (!empty($order_details->use_calculated_delivery_tax) && $order_details->use_calculated_delivery_tax) {
            $this->_AddElement($order, 'UseCalculatedDeliveryTax', 'true');
        }
        if (!empty($order_details->use_calculated_dissur_tax) && $order_details->use_calculated_dissur_tax) {
            $this->_AddElement($order, 'UseCalculatedDissurTax', 'true');
        }

        // Add the order analysis
        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($order_details->analysis['sales_order'])) {
            $analysis = $this->_xml->dom->createElement('SalesOrderAnalysis');
            $order->appendChild($analysis);
            foreach ($order_details->analysis['sales_order'] as $k => $v) {
                $this->_AddElement($analysis, strtoupper($k), $v);
            }
        }

        // Add the customer node to the order
        $customer = $this->_xml->dom->createElement('Customer');
        $order->appendChild($customer);
        $this->_AddElement($customer, 'AccountNumber', $order_details->customer_account);
        $this->_AddElement($customer, 'eCommerceAccountNumber', $order_details->customer_account);
        $this->_AddElement($customer, 'CurrencyCode', $order_details->customer_currency_code);
        $this->_AddElement($customer, 'TaxCode', $order_details->customer_tax_code);
        $this->_AddElement($customer, 'VatNumber', $order_details->customer_vat_number);

        $name = (!empty($order_details->customer['statement_addr']->company) && $this->_use_company_name_as_statement_name) ? $order_details->customer['statement_addr']->company : $order_details->customer['statement_addr']->name;
        $this->_AddElement($customer, 'StatementName', $name);
        $this->_AddElement($customer, 'StatementAddress1', $order_details->customer['statement_addr']->line1);
        $this->_AddElement($customer, 'StatementAddress2', $order_details->customer['statement_addr']->line2);
        $this->_AddElement($customer, 'StatementAddress3', $order_details->customer['statement_addr']->line3);
        $this->_AddElement($customer, 'StatementTown', $order_details->customer['statement_addr']->town);
        $this->_AddElement($customer, 'StatementCounty', $order_details->customer['statement_addr']->county);
        $this->_AddElement($customer, 'StatementCountry', $order_details->customer['statement_addr']->country);
        $this->_AddElement($customer, 'StatementCountryCode', $order_details->customer['statement_addr']->countryiso);
        $this->_AddElement($customer, 'StatementPostcode', $order_details->customer['statement_addr']->postcode);
        $this->_AddElement($customer, 'StatementEmail', $order_details->customer['statement_addr']->email);
        $this->_AddElement($customer, 'StatementWebsite', $order_details->customer['statement_addr']->website);
        $this->_AddElement($customer, 'StatementTelephone', $order_details->customer['statement_addr']->telephone);
        $this->_AddElement($customer, 'StatementFax', $order_details->customer['statement_addr']->fax);

        $name = ($order_details->customer['invoice_addr']->company != '') ? $order_details->customer['invoice_addr']->company : $order_details->customer['invoice_addr']->name;
        $this->_AddElement($customer, 'InvoiceName', $name);
        $this->_AddElement($customer, 'InvoiceAddress1', $order_details->customer['invoice_addr']->line1);
        $this->_AddElement($customer, 'InvoiceAddress2', $order_details->customer['invoice_addr']->line2);
        $this->_AddElement($customer, 'InvoiceAddress3', $order_details->customer['invoice_addr']->line3);
        $this->_AddElement($customer, 'InvoiceTown', $order_details->customer['invoice_addr']->town);
        $this->_AddElement($customer, 'InvoiceCounty', $order_details->customer['invoice_addr']->county);
        $this->_AddElement($customer, 'InvoiceCountry', $order_details->customer['invoice_addr']->country);
        $this->_AddElement($customer, 'InvoiceCountryCode', $order_details->customer['invoice_addr']->countryiso);
        $this->_AddElement($customer, 'InvoicePostcode', $order_details->customer['invoice_addr']->postcode);
        $this->_AddElement($customer, 'InvoiceEmail', $order_details->customer['invoice_addr']->email);
        $this->_AddElement($customer, 'InvoiceWebsite', $order_details->customer['invoice_addr']->website);
        $this->_AddElement($customer, 'InvoiceTelephone', $order_details->customer['invoice_addr']->telephone);
        $this->_AddElement($customer, 'InvoiceFax', $order_details->customer['invoice_addr']->fax);

        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($order_details->analysis['customer'])) {
            $analysis = $this->_xml->dom->createElement('Analysis');
            $customer->appendChild($analysis);
            foreach ($order_details->analysis['customer'] as $k => $v) {
                $this->_AddElement($analysis, strtoupper($k), $v);
            }
        }

        // Add the customer contact node to the order
        $customer_contact_salutation = ($order_details->customer['contact']->salutation == '') ? $order_details->customer['statement_addr']->salutation : $order_details->customer['contact']->salutation;
        $customer_contact_name = ($order_details->customer['contact']->name == '') ? $order_details->customer['statement_addr']->name : $order_details->customer['contact']->name;
        $customer_contact_position = ($order_details->customer['contact']->position == '') ? $order_details->customer['statement_addr']->position : $order_details->customer['contact']->position;
        $customer_contact_telephone = ($order_details->customer['contact']->telephone == '') ? $order_details->customer['statement_addr']->telephone : $order_details->customer['contact']->telephone;
        $customer_contact_fax = ($order_details->customer['contact']->fax == '') ? $order_details->customer['statement_addr']->fax : $order_details->customer['contact']->fax;
        $customer_contact_email = ($order_details->customer['contact']->email == '') ? $order_details->customer['statement_addr']->email : $order_details->customer['contact']->email;
        $customer_contact_mobile = ($order_details->customer['contact']->mobile == '') ? $order_details->customer['statement_addr']->mobile : $order_details->customer['contact']->mobile;
        $customer_contact_extension = ($order_details->customer['contact']->extension == '') ? $order_details->customer['statement_addr']->extension : $order_details->customer['contact']->extension;
        $contact = $this->_xml->dom->createElement('CustomerContact');
        $customer->appendChild($contact);
        $this->_AddElement($contact, 'Salutation', $customer_contact_salutation);
        $this->_AddElement($contact, 'Name', $customer_contact_name);
        $this->_AddElement($contact, 'Position', $customer_contact_position);
        $this->_AddElement($contact, 'Telephone', $customer_contact_telephone);
        $this->_AddElement($contact, 'Fax', $customer_contact_fax);
        $this->_AddElement($contact, 'Email', $customer_contact_email);
        $this->_AddElement($contact, 'Mobile', $customer_contact_mobile);
        $this->_AddElement($contact, 'Extension', $customer_contact_extension);
        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($order_details->analysis['contact'])) {
            $analysis = $this->_xml->dom->createElement('Analysis');
            $contact->appendChild($analysis);
            foreach ($order_details->analysis['contact'] as $k => $v) {
                $this->_AddElement($analysis, strtoupper($k), $v);
            }
        }

        // Add the delivery address node to the customer
        $del_address = $this->_xml->dom->createElement('DeliveryAddress');
        $customer->appendChild($del_address);
        $delivery_name = ($order_details->customer['delivery_addr']->name == '') ? $order_details->customer['delivery_addr']->contact : $order_details->customer['delivery_addr']->name;
        $delivery_contact = ($delivery_name == $order_details->customer['delivery_addr']->contact) && !$this->_allow_matching_delivery_contact_name ? '' : $order_details->customer['delivery_addr']->contact;
        $this->_AddElement($del_address, 'Name', $delivery_name);
        $this->_AddElement($del_address, 'Contact', $delivery_contact);
        $this->_AddElement($del_address, 'Address1', $order_details->customer['delivery_addr']->line1);
        $this->_AddElement($del_address, 'Address2', $order_details->customer['delivery_addr']->line2);
        $this->_AddElement($del_address, 'Address3', $order_details->customer['delivery_addr']->line3);
        $this->_AddElement($del_address, 'Town', $order_details->customer['delivery_addr']->town);
        $this->_AddElement($del_address, 'County', $order_details->customer['delivery_addr']->county);
        $this->_AddElement($del_address, 'Country', $order_details->customer['delivery_addr']->country);
        $this->_AddElement($del_address, 'CountryCode', $order_details->customer['delivery_addr']->countryiso);
        $this->_AddElement($del_address, 'Postcode', $order_details->customer['delivery_addr']->postcode);
        $this->_AddElement($del_address, 'Email', $order_details->customer['delivery_addr']->email);
        $this->_AddElement($del_address, 'Website', $order_details->customer['delivery_addr']->website);
        $this->_AddElement($del_address, 'Telephone', $order_details->customer['delivery_addr']->telephone);
        $this->_AddElement($del_address, 'Fax', $order_details->customer['delivery_addr']->fax);
        $this->_AddElement($del_address, 'ANANumber', $order_details->customer['delivery_addr']->ana_number);
        $this->_AddElement($del_address, 'SpecialInstructions', $order_details->customer['delivery_addr']->instructions);
        $this->_AddElement($del_address, 'EU', $order_details->customer['delivery_addr']->eu);
        $this->_AddElement($del_address, 'Export', $order_details->customer['delivery_addr']->export);
        $this->_AddElement($del_address, 'DeliveryGroup', $order_details->customer['delivery_addr']->delivery_group);
        $this->_AddElement($del_address, 'DeliveryMethod', $order_details->customer['delivery_addr']->delivery_method);
        $this->_AddElement($del_address, 'DefaultAddress', $order_details->customer['delivery_addr']->default_address);
        $this->_AddElement($del_address, 'DeliveryRoute', $order_details->customer['delivery_addr']->delivery_route);
        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($order_details->analysis['delivery'])) {
            $analysis = $this->_xml->dom->createElement('Analysis');
            $del_address->appendChild($analysis);
            foreach ($order_details->analysis['delivery'] as $k => $v) {
                $this->_AddElement($analysis, strtoupper($k), $v);
            }
        }

        // Order financials, delivery and special instructions
        $this->_AddElement($order, 'DeliveryMethod', $order_details->delivery_method);
        $this->_AddElement($order, 'DeliveryNet', round((float)$order_details->delivery_net, 4));
        $this->_AddElement($order, 'DeliveryTax', round((float)$order_details->delivery_vat, 4));
        $this->_AddElement($order, 'DeliveryGross', round((float)$order_details->delivery_gross, 2));
        if ($order_details->delivery_vat_code != '') {
            $this->_AddElement($order, 'DeliveryTaxCode', $order_details->delivery_vat_code);
        }
        $this->_AddElement($order, 'OrderNet', round((float)$order_details->order_net, 4));
        $this->_AddElement($order, 'OrderTax', round((float)$order_details->order_vat, 4));
        $this->_AddElement($order, 'OrderGross', round((float)$order_details->order_gross, 2));

        // Add the payment nodes
        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($order_details->payments)) {
            $payments = $this->_xml->dom->createElement('Payments');
            $amount_paid = 0;
            $order->appendChild($payments);
            foreach ($order_details->payments as $payment) {
                if ((float)$payment['amount'] > 0) {
                    $sales_payment = $this->_xml->dom->createElement('SalesPayment');
                    $payments->appendChild($sales_payment);
                    $amt = round((float)$payment['amount'], 2);
                    $this->_AddElement($sales_payment, 'Description', $payment['description']);
                    $this->_AddElement($sales_payment, 'Amount', $amt);
                    $this->_AddElement($sales_payment, 'Comments', $payment['comment']);
                    $this->_AddElement($sales_payment, 'InternetPaymentRef', $payment['internet_payment_ref']);
                    // 21/07/2020 | CB | Dev 50534 | Added
                    $this->_AddElement($sales_payment, 'InternetPaymentAuthCode', $payment['internet_payment_auth_code']);
                    $this->_AddElement($sales_payment, 'InternetPaymentAdditionalRef', $payment['internet_payment_additional_ref']);
                    $amount_paid += $amt;

                    if (isset($payment['vanguard_transaction_token'])) {
                        $vanguard = $this->_xml->dom->createElement('VanguardDetails');
                        $sales_payment->appendChild($vanguard);
                        $this->_AddElement($vanguard, 'TransactionToken', $payment['vanguard_transaction_token']);
                        $this->_AddElement($vanguard, 'TransactionDB', $payment['vanguard_transaction_db']);
                        $this->_AddElement($vanguard, 'TransactionExpiry', $payment['vanguard_transaction_expiry']);
                        $this->_AddElement($vanguard, 'PreAuthPayment', $payment['vanguard_preauth_payment']);
                        $this->_AddElement($vanguard, 'MerchantRef', $payment['vanguard_merchant_ref']);
                        $this->_AddElement($vanguard, 'PayerAuthRequestID', $payment['vanguard_payer_auth_request_id']);
                        $this->_AddElement($vanguard, 'PayerAuthStatus', $payment['vanguard_payer_auth_status']);
                        $this->_AddElement($vanguard, 'PayerAuthCAVV', $payment['vanguard_payer_auth_cavv']);
                        $this->_AddElement($vanguard, 'PayerAuthECI', $payment['vanguard_payer_auth_eci']);
                        $this->_AddElement($vanguard, 'PayerAuthATSData', $payment['vanguard_payer_auth_atsdata']);
                        $this->_AddElement($vanguard, "AuthCode", $payment["vanguard_auth_code"]);
                    }
                }
            }
            $this->_AddElement($order, 'AmountPaid', round($amount_paid, 2));
        }

        // Add the new collection nodes
        // 12/07/2017 TM | If an empty CollectionAddress is passed OrderWise errors as it expects data and doesn't correctly treat it as empty
        if (isset($order_details->collection)) {
            $collection = $this->_xml->dom->createElement('CollectionAddress');
            $order->appendChild($collection);
            $this->_AddElement($collection, 'Name', $order_details->collection['name']);
            $this->_AddElement($collection, 'Address1', $order_details->collection['address1']);
            $this->_AddElement($collection, 'Address2', $order_details->collection['address2']);
            $this->_AddElement($collection, 'Address3', $order_details->collection['address3']);
            $this->_AddElement($collection, 'Town', $order_details->collection['town']);
            $this->_AddElement($collection, 'County', $order_details->collection['county']);
            $this->_AddElement($collection, 'Country', $order_details->collection['country']);
            $this->_AddElement($collection, 'CountryCode', $order_details->collection['country_code']);
            $this->_AddElement($collection, 'Postcode', $order_details->collection['postcode']);
            $this->_AddElement($collection, 'LocationCode', $order_details->collection['location_code']);
        }

        // Add the discount nodes
        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($order_details->discounts)) {
            $dissurs = $this->_xml->dom->createElement('Dissurs');
            $order->appendChild($dissurs);
            foreach ($order_details->discounts as $dissur) {
                if ((float)$dissur['price'] > 0 && $dissur['description'] != '') {
                    $sales_dissur = $this->_xml->dom->createElement('SalesDissur');
                    $dissurs->appendChild($sales_dissur);
                    $this->_AddElement($sales_dissur, 'Description', $dissur['description']);
                    $this->_AddElement($sales_dissur, 'Price', round((float)$dissur['price'], 4));
                    $this->_AddElement($sales_dissur, 'TaxCode', $dissur['tax_code']);
                    if ($dissur['gross_discount']) {
                        $this->_AddElement($sales_dissur, 'GrossDiscount', 'true');
                    }
                }
            }
        }

        // Create the product nodes
        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($order_details->order_lines)) {
            $_prices_as_net = false;
            $lines = $this->_xml->dom->createElement('Lines');
            $order->appendChild($lines);
            foreach ($order_details->order_lines as $order_line) {
                $line = $this->_xml->dom->createElement('SalesOrderLine');
                $lines->appendChild($line);
                $this->_AddElement($line, 'eCommerceCode', $order_line['ecommerce_code']);
                $this->_AddElement($line, 'Code', $order_line['variant_code']);
                $this->_AddElement($line, 'CustomerPartNumber', $order_line['customer_part_number']);
                $this->_AddElement($line, 'EANCode', $order_line['ean']);
                $this->_AddElement($line, 'Quantity', $order_line['quantity']);
                $this->_AddElement($line, 'eCommerceItemID', $order_line['external_item_id']);
                $this->_AddElement($line, 'ProductComments', $order_line['product_comments']);
                $this->_AddElement($line, 'Comments1', $order_line['comments1']);
                $this->_AddElement($line, 'Comments2', $order_line['comments2']);
                $this->_AddElement($line, 'Description', $order_line['product_description']);
                $this->_AddElement($line, 'ProductComments', $order_line['product_comments']);
                $this->_AddElement($line, 'VariantComments', $order_line['variant_comments']);
                $this->_AddElement($line, 'RSP', $order_line['rsp']);
				
                // 12/06/2020 | CB | Dev 49537 | Added
                if (!empty($order_line['date_promised']) && strtotime($order_line['date_promised']) != null) {
                    $this->_AddElement($line, 'DatePromised', date('Y-m-d\TH:i:s', strtotime($order_line['date_promised'])));
                }

                //22/05/2019 | TL | Dev 43368 | Script Edit to handle multiple stock locations | Adding Line Stock Location element.
                $this->_AddElement($line, 'StockLocation', $order_line['stock_location_id']);

                // 05/03/2019 | DS | Dev 40773 | Added direct XML line for direct order shipments
                if (isset($order_line["direct"])) {
                    $this->_AddElement($line, 'Direct', $order_line['direct'] ? 'true' : 'false');
                }
                if (isset($order_line["item_net"])) {
                    $this->_AddElement($line, 'ItemNet', round((float)$order_line['item_net'], 4));
                    $_prices_as_net = true;
                } else if (isset($order_line["item_gross"])) {
                        $this->_AddElement($line, 'ItemGross', round((float)$order_line['item_gross'], 2));
                }
                $this->_AddElement($line, 'TaxCode', $order_line['vat_code']);

                if (is_array($order_line['line_analysis'])) {
                    $line_analysis = $this->_xml->dom->createElement('Analysis');
                    $line->appendChild($line_analysis);
                    foreach ($order_line['line_analysis'] as $k => $v) {
                        $this->_AddElement($line_analysis, strtoupper($k), $v);
                    }
                }
                if (is_array($order_line['variant_analysis'])) {
                    $variant_analysis = $this->_xml->dom->createElement('VariantAnalysis');
                    $line->appendChild($variant_analysis);
                    foreach ($order_line['variant_analysis'] as $k => $v) {
                        $this->_AddElement($variant_analysis, strtoupper($k), $v);
                    }
                }
                if (is_array($order_line['product_analysis'])) {
                    $product_analysis = $this->_xml->dom->createElement('ProductAnalysis');
                    $line->appendChild($product_analysis);
                    foreach ($order_line['product_analysis'] as $k => $v) {
                        $this->_AddElement($product_analysis, strtoupper($k), $v);
                    }
                }
                // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
                if (!empty($order_line['attachments'])) {
                    $attachments = $this->_xml->dom->createElement('Attachments');
                    $line->appendChild($attachments);
                    foreach ($order_line['attachments'] as $attach) {
                        $sol_attachment = $this->_xml->dom->createElement('SalesOrderLineAttachment');
                        $attachments->appendChild($sol_attachment);
                        $this->_AddElement($sol_attachment, 'Attachment', $attach['attachment']);
                        $this->_AddElement($sol_attachment, 'Description', $attach['description']);
                        $this->_AddElement($sol_attachment, 'Notes', $attach['notes']);
                        $this->_AddElement($sol_attachment, 'Image', $attach['image']);
                    }
                }
                // 03/02/2017 | CB | Dev 21264 | Added components
                // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
                if (!empty($order_line['components'])) {
                    $components = $this->_xml->dom->createElement('Components');
                    $line->appendChild($components);
                    foreach ($order_line['components'] as $component) {
                        $sol_component = $this->_xml->dom->createElement('SalesOrderLineComponent');
                        $components->appendChild($sol_component);
                        $this->_AddElement($sol_component, 'Code', $component['code']);
                        $this->_AddElement($sol_component, 'Quantity', $component['qty']);
                        $this->_AddElement($sol_component, 'BuildPer', $component['build_per']);
                        if (is_array($component['component_analysis']) && !empty($component['component_analysis'])) {
                            $component_analysis = $this->_xml->dom->createElement('Analysis');
                            $sol_component->appendChild($component_analysis);
                            foreach ($component['component_analysis'] as $k => $v) {
                                $this->_AddElement($component_analysis, strtoupper($k), $v);
                            }
                        }
                    }
                }

                // 25/07/2018 | CB | Dev 37647 | Added Traceability nodes
                // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
                if (!empty($order_line['traceability'])) {
                    $traceability_list = $this->_xml->dom->createElement('TraceabilityList');
                    $line->appendChild($traceability_list);
                    foreach ($order_line['traceability'] as $traceability) {
                        $sol_traceability = $this->_xml->dom->createElement('Traceability');
                        $traceability_list->appendChild($sol_traceability);
                        $this->_AddElement($sol_traceability, 'BatchNumber', $traceability['batch_number']);
                        $this->_AddElement($sol_traceability, 'Quantity', $traceability['qty']);
                    }
                }
            }

            // If we found net prices then flag the order to be PricesAsNet
            if ($_prices_as_net || $order_details->set_prices_as_net_flag) {
                $this->_AddElement($order, 'PricesAsNet', 'true');
            }
        }

        // Increments the order counter so that should there be a limit
        // to the number of orders we import we can detect and drop out
        $this->SetOrderCounter($this->GetOrderCounter() + 1);
        return true;
    }

    /**
     * Adds an element to an xml document
     *
     * @param object $parent
     * @param object $node
     * @param string $value
     */
    private function _AddElement(&$parent, $node, $value = '')
    {
        $value = trim($value);
        if ($this->_mbstring_available) {
            if (!mb_detect_encoding($value, 'UTF-8', true)) {
                $value = utf8_encode($value);
            }
        }
        if ($value == '') {
            return;
        }

        if (preg_match('/[\<\>\&\'\"\/]/i', $value)) {
            $e = $this->_xml->dom->createElement($node);
            $e->appendChild($this->_xml->dom->createCDATASection($value));
            $parent->appendChild($e);
        } else {
            $parent->appendChild($this->_xml->dom->createElement($node, $value));
        }
    }

    /**
     * @return int
     */
    public function GetOrderCounter()
    {
        return $this->_order_counter;
    }

    /**
     * @param $v
     */
    public function SetOrderCounter($v)
    {
        $this->_order_counter = $v;
    }

    /**
     * Builds the v7 XML customer import data
     *
     * @param object $customer_details
     * @param bool
     */
    public function AppendCustomerToXml($customer_details)
    {
        $customer = $this->_xml->dom->createElement('Customer');
        $this->_xml->customers->appendChild($customer);

        // General account details
        $this->_AddElement($customer, 'eCommerceAccountNumber', $customer_details->customer_account);
        $this->_AddElement($customer, 'VatNumber', $customer_details->customer_vat_number);

        // Statement address
        $statement_address = $this->_xml->dom->createElement('StatementAddress');
        $customer->appendChild($statement_address);
        $name = ($customer_details->customer['statement_addr']->company != '') ? $customer_details->customer['statement_addr']->company : $customer_details->customer['statement_addr']->name;
        $this->_AddElement($statement_address, 'StatementName', $name);
        $this->_AddElement($statement_address, 'StatementAddress1', $customer_details->customer['statement_addr']->line1);
        $this->_AddElement($statement_address, 'StatementAddress2', $customer_details->customer['statement_addr']->line2);
        $this->_AddElement($statement_address, 'StatementAddress3', $customer_details->customer['statement_addr']->line3);
        $this->_AddElement($statement_address, 'StatementTown', $customer_details->customer['statement_addr']->town);
        $this->_AddElement($statement_address, 'StatementCounty', $customer_details->customer['statement_addr']->county);
        $this->_AddElement($statement_address, 'StatementCountry', $customer_details->customer['statement_addr']->country);
        $this->_AddElement($statement_address, 'StatementCountryCode', $customer_details->customer['statement_addr']->countryiso);
        $this->_AddElement($statement_address, 'StatementPostcode', $customer_details->customer['statement_addr']->postcode);
        $this->_AddElement($statement_address, 'StatementEmail', $customer_details->customer['statement_addr']->email);
        $this->_AddElement($statement_address, 'StatementWebsite', $customer_details->customer['statement_addr']->website);
        $this->_AddElement($statement_address, 'StatementTelephone', $customer_details->customer['statement_addr']->telephone);
        $this->_AddElement($statement_address, 'StatementFax', $customer_details->customer['statement_addr']->fax);

        // Customer contact
        $contact = $this->_xml->dom->createElement('CustomerContact');
        $customer->appendChild($contact);
        $this->_AddElement($contact, 'Salutation', $customer_details->customer['contact']->salutation);
        $this->_AddElement($contact, 'Name', $customer_details->customer['contact']->name);
        $this->_AddElement($contact, 'Position', $customer_details->customer['contact']->position);
        $this->_AddElement($contact, 'Telephone', $customer_details->customer['contact']->telephone);
        $this->_AddElement($contact, 'Fax', $customer_details->customer['contact']->fax);
        $this->_AddElement($contact, 'Email', $customer_details->customer['contact']->email);
        $this->_AddElement($contact, 'Mobile', $customer_details->customer['contact']->mobile);
        $this->_AddElement($contact, 'Extension', $customer_details->customer['contact']->extension);
    }

    /**
     * Exports the order to a file rather than the dom. Used to create
     * individual files from a file containing multiple orders eg a CVS file
     *
     * @param $order_id
     * @param string $output_dir
     * @return bool
     */
    public function ExportOrderToFile($order_id, $output_dir = '')
    {
        $format = new DataFormat();
        $output_dir = ($output_dir == '') ? $this->GetDirectory('downloads_pending') : $output_dir;
        if (file_put_contents($output_dir . $format->StrToSafeChars($order_id) . '.xml', $this->_FormatXmlOutput())) {
            return true;
        }
        return false;
    }

    /**
     * Formats the XML data being returned to OrderWise
     *
     * $this->_xml->dom
     */
    private function _FormatXmlOutput()
    {
        $x = new DOMDocument('1.0', 'UTF-8');
        $x->preserveWhiteSpace = false;
        $x->formatOutput = true;
        $x->loadXML($this->_xml->dom->saveXML());
        return $x->saveXML();
    }

    /**
     * Returns XML data to OrderWise
     */
    public function ExportToXml()
    {
        // Create the response element
        // 12/02/2019 | DS | PHP 7.2+ does not allowing count to be used on non-arrays (null variables/objects)
        if (!empty($this->_xml_response_messages)) {
            foreach ($this->_xml_response_messages as $k => $v) {
                $response = $this->_xml->dom->createElement('Response');
                $this->_xml->reponses->appendChild($response);
                $this->_AddElement($response, 'Type', $v['type']);
                $this->_AddElement($response, 'Message', '[WebScript] ' . $v['message']);
            }
        }
        clearstatcache();
        header('content-type: text/xml; charset=utf-8');
        header('Cache-Control: no-cache');
        header('Pragma: no-cache');
        echo $this->_FormatXmlOutput();
    }

    /**
     * @param $curl_data
     * @param $response
     */
    public function CurlExec($curl_data, &$response)
    {
        $ch = curl_init($curl_data['endpoint']);

        // Set the defaults for a basic connection
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HEADER, false);

        // Override the defaults
        if (is_array($curl_data['config'])) {
            foreach ($curl_data['config'] as $k => $v) {
                curl_setopt($ch, $k, $v);
            }
        }
        $r = curl_exec($ch);

        // Run the curl request
        $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $response = [
            'content' => $r,
            'header' => substr($r, 0, $header_size),
            'body' => substr($r, $header_size),
            'http_code' => curl_getinfo($ch, CURLINFO_HTTP_CODE),
            'header_size' => $header_size,
        ];

        curl_close($ch);
    }

    /**
     * Sets the database to be used by these methods and any extended classes
     *
     * @param object $v
     */
    public function SetDatabase($v)
    {
        $this->_db = $v;
    }

    /**
     * @param $v
     * @return mixed
     */
    public function SetXmlDom($v)
    {
        return $this->_xml = $v;
    }

    /**
     * @param $n
     * @param $p
     */
    public function SetDirectory($n, $p)
    {
        $this->_config->directories[$n] = $p;
    }

    /**
     * @param $v
     */
    public function SetStoreId($v)
    {
        $this->_config->store_id = $v;
    }

    /**
     * @return mixed
     */
    public function GetDbMode()
    {
        return $this->_config->db_mode;
    }

    /**
     * @return mixed
     */
    public function GetXmlDom()
    {
        return $this->_xml;
    }

    /**
     * @return mixed
     */
    public function GetMaximumOrdersPerSession()
    {
        return $this->_config->maximum_orders_per_session;
    }


    /**
     * @return mixed
     */
    public function GetSkuFilterCount()
    {
        return $this->_config->sku_filter_cnt;
    }


    /**
     * @return mixed
     */
    public function GetSkuFilterList()
    {
        return $this->_config->sku_filter_list;
    }
}