<?php

class DataFormat
{
    private $_mbstring_available = false;
    private $_isotocountry = array();
    private $_countrytoiso = array();


    /**
     * Check whether the mbstring extension is loaded
     */
    function __construct()
    {
        $this->_SetCountryISO();
        if (extension_loaded('mbstring')) {
            mb_internal_encoding('UTF-8');
            $this->_mbstring_available = true;
        }
    }

    /**
     * Sets an array of country >> iso and iso >> country code values.
     *
     */
    private function _SetCountryISO()
    {
        $arr = array(
            'AC' => 'Ascension Island',
            'AD' => 'Andorra',
            'AE' => 'United Arab Emirates',
            'AF' => 'Afghanistan',
            'AG' => 'Antigua and Barbuda',
            'AI' => 'Anguilla',
            'AL' => 'Albania',
            'AM' => 'Armenia',
            'AN' => 'Netherlands Antilles',
            'AO' => 'Angola',
            'AQ' => 'Antarctica',
            'AR' => 'Argentina',
            'AS' => 'American Samoa',
            'AT' => 'Austria',
            'AU' => 'Australia',
            'AW' => 'Aruba',
            'AX' => 'Aland Islands',
            'AZ' => 'Azerbaijan',
            'BA' => 'Bosnia and Herzegovina',
            'BB' => 'Barbados',
            'BD' => 'Bangladesh',
            'BE' => 'Belgium',
            'BF' => 'Burkina Faso',
            'BG' => 'Bulgaria',
            'BH' => 'Bahrain',
            'BI' => 'Burundi',
            'BJ' => 'Benin',
            'BM' => 'Bermuda',
            'BN' => 'Brunei Darussalam',
            'BO' => 'Bolivia',
            'BR' => 'Brazil',
            'BS' => 'Bahamas',
            'BT' => 'Bhutan',
            'BV' => 'Bouvet Island',
            'BW' => 'Botswana',
            'BY' => 'Belarus',
            'BZ' => 'Belize',
            'CA' => 'Canada',
            'CC' => 'Cocos (Keeling) Islands',
            'CD' => 'Congo, Democratic Republic',
            'CF' => 'Central African Republic',
            'CG' => 'Congo',
            'CH' => 'Switzerland',
            'CI' => 'Cote D\'Ivoire (Ivory Coast)',
            'CK' => 'Cook Islands',
            'CL' => 'Chile',
            'CM' => 'Cameroon',
            'CN' => 'China',
            'CO' => 'Colombia',
            'CR' => 'Costa Rica',
            'CS' => 'Czechoslovakia (former)',
            'CU' => 'Cuba',
            'CV' => 'Cape Verde',
            'CX' => 'Christmas Island',
            'CY' => 'Cyprus',
            'CZ' => 'Czech Republic',
            'DE' => 'Germany',
            'DJ' => 'Djibouti',
            'DK' => 'Denmark',
            'DM' => 'Dominica',
            'DO' => 'Dominican Republic',
            'DZ' => 'Algeria',
            'EC' => 'Ecuador',
            'EE' => 'Estonia',
            'EG' => 'Egypt',
            'EH' => 'Western Sahara',
            'ER' => 'Eritrea',
            'ES' => 'Spain',
            'ET' => 'Ethiopia',
            'EU' => 'European Union',
            'FI' => 'Finland',
            'FJ' => 'Fiji',
            'FK' => 'Falkland Islands (Malvinas)',
            'FM' => 'Micronesia',
            'FO' => 'Faroe Islands',
            'FR' => 'France',
            'FX' => 'France, Metropolitan',
            'GA' => 'Gabon',
            'GB' => 'United Kingdom',
            'GD' => 'Grenada',
            'GE' => 'Georgia',
            'GF' => 'French Guiana',
            'GG' => 'Guernsey',
            'GH' => 'Ghana',
            'GI' => 'Gibraltar',
            'GL' => 'Greenland',
            'GM' => 'Gambia',
            'GN' => 'Guinea',
            'GP' => 'Guadeloupe',
            'GQ' => 'Equatorial Guinea',
            'GR' => 'Greece',
            'GS' => 'S. Georgia and S. Sandwich Isls.',
            'GT' => 'Guatemala',
            'GU' => 'Guam',
            'GW' => 'Guinea-Bissau',
            'GY' => 'Guyana',
            'HK' => 'Hong Kong',
            'HM' => 'Heard and McDonald Islands',
            'HN' => 'Honduras',
            'HR' => 'Croatia (Hrvatska)',
            'HT' => 'Haiti',
            'HU' => 'Hungary',
            'ID' => 'Indonesia',
            'IE' => 'Ireland',
            'IL' => 'Israel',
            'IM' => 'Isle of Man',
            'IN' => 'India',
            'IO' => 'British Indian Ocean Territory',
            'IQ' => 'Iraq',
            'IR' => 'Iran',
            'IS' => 'Iceland',
            'IT' => 'Italy',
            'JE' => 'Jersey',
            'JM' => 'Jamaica',
            'JO' => 'Jordan',
            'JP' => 'Japan',
            'KE' => 'Kenya',
            'KG' => 'Kyrgyzstan',
            'KH' => 'Cambodia',
            'KI' => 'Kiribati',
            'KM' => 'Comoros',
            'KN' => 'Saint Kitts and Nevis',
            'KP' => 'Korea (North)',
            'KR' => 'Korea (South)',
            'KW' => 'Kuwait',
            'KY' => 'Cayman Islands',
            'KZ' => 'Kazakhstan',
            'LA' => 'Laos',
            'LB' => 'Lebanon',
            'LC' => 'Saint Lucia',
            'LI' => 'Liechtenstein',
            'LK' => 'Sri Lanka',
            'LR' => 'Liberia',
            'LS' => 'Lesotho',
            'LT' => 'Lithuania',
            'LU' => 'Luxembourg',
            'LV' => 'Latvia',
            'LY' => 'Libya',
            'MA' => 'Morocco',
            'MC' => 'Monaco',
            'MD' => 'Moldova',
            'ME' => 'Montenegro',
            'MF' => 'Saint Martin',
            'MG' => 'Madagascar',
            'MH' => 'Marshall Islands',
            'MK' => 'F.Y.R.O.M. (Macedonia)',
            'ML' => 'Mali',
            'MM' => 'Myanmar',
            'MN' => 'Mongolia',
            'MO' => 'Macau',
            'MP' => 'Northern Mariana Islands',
            'MQ' => 'Martinique',
            'MR' => 'Mauritania',
            'MS' => 'Montserrat',
            'MT' => 'Malta',
            'MU' => 'Mauritius',
            'MV' => 'Maldives',
            'MW' => 'Malawi',
            'MX' => 'Mexico',
            'MY' => 'Malaysia',
            'MZ' => 'Mozambique',
            'NA' => 'Namibia',
            'NC' => 'New Caledonia',
            'NE' => 'Niger',
            'NF' => 'Norfolk Island',
            'NG' => 'Nigeria',
            'NI' => 'Nicaragua',
            'NL' => 'Netherlands',
            'NO' => 'Norway',
            'NP' => 'Nepal',
            'NR' => 'Nauru',
            'NT' => 'Neutral Zone',
            'NU' => 'Niue',
            'NZ' => 'New Zealand (Aotearoa)',
            'OM' => 'Oman',
            'PA' => 'Panama',
            'PE' => 'Peru',
            'PF' => 'French Polynesia',
            'PG' => 'Papua New Guinea',
            'PH' => 'Philippines',
            'PK' => 'Pakistan',
            'PL' => 'Poland',
            'PM' => 'St. Pierre and Miquelon',
            'PN' => 'Pitcairn',
            'PR' => 'Puerto Rico',
            'PS' => 'Palestinian Territory, Occupied',
            'PT' => 'Portugal',
            'PW' => 'Palau',
            'PY' => 'Paraguay',
            'QA' => 'Qatar',
            'RE' => 'Reunion',
            'RS' => 'Serbia',
            'RO' => 'Romania',
            'RU' => 'Russian Federation',
            'RW' => 'Rwanda',
            'SA' => 'Saudi Arabia',
            'SB' => 'Solomon Islands',
            'SC' => 'Seychelles',
            'SD' => 'Sudan',
            'SE' => 'Sweden',
            'SG' => 'Singapore',
            'SH' => 'St. Helena',
            'SI' => 'Slovenia',
            'SJ' => 'Svalbard & Jan Mayen Islands',
            'SK' => 'Slovak Republic',
            'SL' => 'Sierra Leone',
            'SM' => 'San Marino',
            'SN' => 'Senegal',
            'SO' => 'Somalia',
            'SR' => 'Suriname',
            'ST' => 'Sao Tome and Principe',
            'SU' => 'USSR (former)',
            'SV' => 'El Salvador',
            'SY' => 'Syria',
            'SZ' => 'Swaziland',
            'TC' => 'Turks and Caicos Islands',
            'TD' => 'Chad',
            'TF' => 'French Southern Territories',
            'TG' => 'Togo',
            'TH' => 'Thailand',
            'TJ' => 'Tajikistan',
            'TK' => 'Tokelau',
            'TM' => 'Turkmenistan',
            'TN' => 'Tunisia',
            'TO' => 'Tonga',
            'TP' => 'East Timor',
            'TR' => 'Turkey',
            'TT' => 'Trinidad and Tobago',
            'TV' => 'Tuvalu',
            'TW' => 'Taiwan',
            'TZ' => 'Tanzania',
            'UA' => 'Ukraine',
            'UG' => 'Uganda',
            'UM' => 'US Minor Outlying Islands',
            'US' => 'United States',
            'UY' => 'Uruguay',
            'UZ' => 'Uzbekistan',
            'VA' => 'Vatican City State (Holy See)',
            'VC' => 'Saint Vincent & the Grenadines',
            'VE' => 'Venezuela',
            'VG' => 'British Virgin Islands',
            'VI' => 'Virgin Islands (U.S.)',
            'VN' => 'Viet Nam',
            'VU' => 'Vanuatu',
            'WF' => 'Wallis and Futuna Islands',
            'WS' => 'Samoa',
            'XK' => 'Kosovo',
            'YE' => 'Yemen',
            'YT' => 'Mayotte',
            'YU' => 'Serbia and Montenegro (former Yugoslavia)',
            'ZA' => 'South Africa',
            'ZM' => 'Zambia',
            'ZR' => 'Zaire',
            'ZW' => 'Zimbabwe');
        $this->_isotocountry = $arr;
        foreach ($this->_isotocountry as $k => $v) {
            $v = strtolower($v);
            $this->_countrytoiso[$v] = $k;
        }
    }

    /**
     * Returns the first non blank value from a list of parameters. Values are checked in the order they're passed.
     *
     * @param string
     * @return string
     */
    public function GetFirstNonBlank()
    {
        $arg_list = func_get_args();
        for ($x = 0; $x < func_num_args(); $x++) {
            $str = trim($arg_list[$x]);
            if ($str != '') {
                return $str;
            }
        }
    }

    /**
     * Formats a name to the correct casing
     *
     * @param string $str
     * @return string
     */
    public function FormatName($str)
    {
        if ($this->_mbstring_available) {
            $str = mb_convert_case(mb_strtolower($str), MB_CASE_TITLE);
        } else {
            $str = ucwords(strtolower($str));
        }

        foreach (array('-', '\'', 'Mc', 'Mac') as $delimiter) {
            if (strpos($str, $delimiter) !== false) {
                $str = implode($delimiter, array_map('ucfirst', explode($delimiter, $str)));
            }
        }

        $arr = explode(' ', $str);
        $new_str = '';
        foreach ($arr as $v) {
            if (!preg_match('/[aeiou]/i', $v)) {
                if ($this->_mbstring_available) {
                    $v = mb_strtoupper($v);
                } else {
                    $v = strtoupper($v);
                }
            }
            $new_str .= $v . ' ';
        }

        // 27/06/2019 | DS | BUG 59313 | Not all UTF-8 characters are supported by XML. Strip away any invalid characters
        $new_str = $this->RemoveInvalidXmlCharacters($new_str);
        return trim(preg_replace('/[ ]{1,}/', ' ', $new_str));
    }

    /**
     * Basic telephone number check. This doesn't validate the format and
     * is just a simple check on the expected characters of 0-9 +.
     *
     * @param string $str
     * @return string
     */
    public function FormatTelephone($str)
    {
        $str = preg_replace('/[^0-9\+ ]/', '', $str);
        $str = preg_replace('/[ ]{1,}/', ' ', $str);
        // 27/06/2019 | DS | BUG 59856 | Not all UTF-8 characters are supported by XML. Strip away any invalid characters
        $str = $this->RemoveInvalidXmlCharacters($str);
        return trim($str);
    }

    /**
     * Formats an address to the correct casing
     *
     * @param string $str
     * @return string
     */
    public function FormatAddress($str)
    {
        if ($this->_mbstring_available) {
            $str = mb_convert_case(mb_strtolower($str), MB_CASE_TITLE);
        } else {
            $str = ucwords(strtolower($str));
        }

        $str = str_replace(',', ', ', $str);
        foreach (array('-', '\'') as $delimiter) {
            if (strpos($str, $delimiter) !== false) {
                $str = implode($delimiter, array_map('ucfirst', explode($delimiter, $str)));
            }
        }

        // 27/06/2019 | DS | BUG 59313 | Not all UTF-8 characters are supported by XML. Strip away any invalid characters
        $str = $this->RemoveInvalidXmlCharacters($str);
        $str = str_replace(array("\n\r", "\n", "\r"), ', ', $str);
        return trim(preg_replace('/[ ]{1,}/', ' ', $str));
    }

    /**
     * Unserialization fix to handle incorrectly serialised data
     *
     * @param string $str
     * @return string
     */
    public function MbUnserialize($str)
    {
        $str = preg_replace('!s:(\d+):"(.*?)";!se', "'s:'.strlen('$2').':\"$2\";'", $str);
        return unserialize($str);
    }

    /**
     * Basic postcode formatting. This doesn't validate the format, just that the character types are correct.
     *
     * @param string $str
     * @return string
     */
    public function FormatPostcode($str)
    {

        // 27/06/2019 | DS | BUG 59313 | Not all UTF-8 characters are supported by XML. Strip away any invalid characters
        $str = $this->RemoveInvalidXmlCharacters($str);
        $str = preg_replace('/[^A-Z0-9 ]/', '', strtoupper($str));
        return trim($str);
    }

    /**
     * @param float $val
     * @param string $t0
     * @param string $t1
     * @return string
     */
    public function GetVatCode($val, $t0 = 'T0', $t1 = 'T1')
    {
        $t = ($val > 0) ? $t1 : $t0;
        return $t;
    }

    /**
     * Gets the ISO for a country from the look up. The look up array is set when the class is first instantiated.
     *
     * @param string $country
     * @return string
     */
    public function GetIsoFromCountry($country)
    {
        $country = strtolower(trim($country));
        if (isset($this->_countrytoiso[$country])) {
            return $this->_countrytoiso[$country];
        }
        return '';
    }

    /**
     * Gets the country name for an ISO from the look up. The look up array is set when the class is first instantiated.
     *
     * @param string $iso
     * @return string
     */
    public function GetCountryFromIso($iso)
    {
        $iso = strtoupper(trim($iso));
        if (isset($this->_isotocountry[$iso])) {
            return $this->_isotocountry[$iso];
        }
        return '';
    }

    /**
     * Remove any invalid XML characters
     * Should be used when an order line doesn't use any of the other
     * format functions to ensure the line is correctly stripped of
     * any XML-breaking characters from the user-input fields.
     * @param $str
     * @return string
     */
    public function RemoveInvalidXmlCharacters($str){
        // 27/06/2019 | DS | Dev 44542 | Not all UTF-8 characters are supported by XML. Strip away any invalid characters
        // Added as a separate function so it can be used when not utilising the other format functions for a line
        // Should be used when handling ANY user-input field
        $str = preg_replace('/[^\x{0009}\x{000a}\x{000d}\x{0020}-\x{D7FF}\x{E000}-\x{FFFD}]+/u', '', $str);
        return $str;
    }

    /**
     * Gets a safe file name. If working with order files rather than a
     * database, we archive the file and need to ensure the file name
     * doesn't contain invalid characters.
     *
     * @param string $str
     * @return    string
     */
    public function StrToSafeChars($str)
    {
        return preg_replace('/[^a-z0-9]/i', '_', $str);
    }


    /**
     * @param $delimiter
     * @param $str
     * @param string $escapeChar
     * @return array
     *
     * Adapted from https://r.je/php-explode-split-with-escape-character
     */
    public function explodeEscaped($str, $delimiter = ',', $escapeChar = '\\')
    {
        $double = "\0\0\0_doub";

        $escaped = "\0\0\0_esc";

        $str = str_replace($escapeChar . $escapeChar, $double, $str);

        $str = str_replace($escapeChar . $delimiter, $escaped, $str);

        $split = array_map('trim', explode($delimiter, $str));

        foreach ($split as &$val) $val = str_replace([$double, $escaped], [$escapeChar, $delimiter], $val);

        return $split;
    }
	
	
	 /**
     * Add a given number of days to a date and return as a date
     *
     * @param $date
     * @param $days_to_add
     * @param bool $exclude_weekends
     * @return false|string
     */
	public function addDays($date, $days_to_add, $exclude_weekends = true)
    {
        $formatted_date = date('Y-m-d H:i:s', strtotime($date));

        for ($i = 0; $i < $days_to_add; $i++) {
            $day = date('N', strtotime("{$formatted_date}+" . ($i + 1) . "days"));
            if ($exclude_weekends && $day > 5) {
                $days_to_add++;
            }
        }

        return date('Y-m-d H:i:s', strtotime( "{$formatted_date} +{$i} days"));
    }
}