<?php
/**
 * Import sales orders into OrderWise
 *
 * System/Version:      Shopify
 * Developer:           Toby Loudon
 * OnTime Ref:          49248
 * Completed:           28/07/2020
 * Production URL:      http://cestevenson-orderwise.net/orderwise_shopify/orders.php
 * Staging URL:
 */

require './config.php';

try {
    $shopify = new Shopify($ecom->GetStoreId(), $settings);
    $shopify->getTaxTitles();

    $delivery_method_lookup = new CsvLookupLoader('delivery_methods.csv');
    $payment_method_lookup = new CsvLookupLoader('payment_methods.csv');
    $stock_location_lookup = new CsvLookupLoader('stock_locations.csv');
} catch (Exception $e) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
    $ecom->ReturnResponseXML();
}

try {
    $order_filter = array('financial_status' => 'paid', 'fulfillment_status' => 'unshipped', 'status' => 'open', 'updated_at_min' => $ecom->GetOrdersFromDate());
    $order_list = $shopify->LoadOrders($order_filter);
} catch (Exception $e) {
    $ecom->AppendResponseMessage(XML_REPONSE_TYPE_ERROR, $e->getMessage());
    $ecom->ReturnResponseXML();
}

$ecom->CreateOrderXmlHeaders();
if (empty($order_list)) {
    $ecom->ExportToXml();
    exit;
}

$format = new DataFormat();
$imported_order_ids = $ecom->GetPreviouslyImportedOrderIds();

foreach ($order_list as $order_row) {
    if (!is_numeric($_GET['order']) && isset($imported_order_ids[trim($order_row->id)])) {
        continue;
    }

    $order_details = $ecom->NewOrder();
    $order_details->order_id = $order_row->id;
    $order_details->order_date = $order_row->created_at;
    $order_details->customer_order_ref = $order_row->order_number;

    // Address handling (billing, shipping, and default address)
    $s_addr = isset($order_row->billing_address) ? $order_row->billing_address : (isset($order_row->customer->default_address) ? $order_row->customer->default_address : $order_row->shipping_address);
    $order_details->customer_currency_code = $order_row->currency;
    $order_details->customer_account = $order_row->customer->id;
    $order_details->customer['statement_addr']->name = $format->FormatName($s_addr->first_name . ' ' . $s_addr->last_name);
    $order_details->customer['statement_addr']->line1 = $format->FormatAddress($s_addr->address1);
    $order_details->customer['statement_addr']->line2 = $format->FormatAddress($s_addr->address2);
    $order_details->customer['statement_addr']->town = str_replace('— Please Select —', '', $format->FormatAddress($s_addr->city));
    $order_details->customer['statement_addr']->county = str_replace('— Please Select —', '', $format->FormatAddress($s_addr->province));
    $order_details->customer['statement_addr']->postcode = $format->FormatPostcode($s_addr->zip);
    $order_details->customer['statement_addr']->country = $s_addr->country;
    $order_details->customer['statement_addr']->countryiso = $s_addr->country_code;
    $order_details->customer['statement_addr']->telephone = $format->FormatTelephone($s_addr->phone);
    $order_details->customer['statement_addr']->email = $order_row->email;
    $order_details->customer['invoice_addr'] = clone $order_details->customer['statement_addr'];

    // Delivery address details
    $d_addr = $order_row->shipping_address;
    $order_details->customer['delivery_addr']->name = $format->FormatName($d_addr->company);
    $order_details->customer['delivery_addr']->contact = $format->FormatName($d_addr->first_name . ' ' . $d_addr->last_name);
    $order_details->customer['delivery_addr']->line1 = $format->FormatAddress($d_addr->address1);
    $order_details->customer['delivery_addr']->line2 = $format->FormatAddress($d_addr->address2);
    $order_details->customer['delivery_addr']->town = str_replace('— Please Select —', '', $format->FormatAddress($d_addr->city));
    $order_details->customer['delivery_addr']->county = str_replace('— Please Select —', '', $format->FormatAddress($d_addr->province));
    $order_details->customer['delivery_addr']->postcode = $format->FormatPostcode($d_addr->zip);
    $order_details->customer['delivery_addr']->country = $d_addr->country;
    $order_details->customer['delivery_addr']->countryiso = $d_addr->country_code;
    $order_details->customer['delivery_addr']->telephone = $format->FormatTelephone($d_addr->phone);
    $order_details->customer['delivery_addr']->email = $order_row->email;

    // Determine VAT rate and tax code based on country and platform
    $is_ireland_order = isset($d_addr->country_code) && strtoupper($d_addr->country_code) == 'IE';
    $is_fruugo_or_onbuy = strpos($order_row->note, 'Fruugo') !== false || strpos($order_row->note, 'OnBuy Marketplace-Integration') !== false;
    $vat_rate = ($is_ireland_order && !$is_fruugo_or_onbuy) ? 0.23 : 0.20;
    $tax_code = ($is_ireland_order && !$is_fruugo_or_onbuy) ? 'T0' : 'T20';

    // Calculate delivery amounts with conditional VAT for Ireland
    $gross_delivery = $order_row->total_shipping_price_set->presentment_money->amount;
    $order_details->delivery_net = $gross_delivery / (1 + $vat_rate);
    $order_details->delivery_vat = $gross_delivery - $order_details->delivery_net;
    $order_details->delivery_gross = $gross_delivery;

    // Process line items with VAT adjustment for Ireland
    foreach ($order_row->line_items as $line) {
        $gross = $line->price;
        $net = $gross / (1 + $vat_rate);
        $line_vat = $gross - $net;

        $order_details->order_lines[] = array(
            'ecommerce_code' => $line->sku,
            'variant_code' => $line->sku,
            'external_item_id' => $line->id,
            'quantity' => $line->quantity,
            'item_net' => $net,
            'item_vat' => $line_vat,
            'item_gross' => $gross,
            'vat_code' => $tax_code
        );
    }

    // Append order to XML
    $ecom->AppendOrderToXml($order_details);
}

$ecom->ExportToXml();
?>
